# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the 'execute_go_ready' skill to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Move to an object location
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. (Continue as needed...)

        # For this example, we will use placeholder object and location names.
        # Replace 'object_1', 'drawer_1', 'location_1', etc. with actual names from your environment.

        # Example object and location names (replace with actual ones)
        object_1 = 'object_1'
        drawer_1 = 'drawer_1'
        location_object = 'location_1'
        location_drawer = 'location_2'
        ready_pose = 'ready-pose'

        # Try to get actual object and location names from positions if available
        if positions:
            # Try to pick the first object and location from the positions dict
            for k, v in positions.items():
                if 'object' in k and object_1 == 'object_1':
                    object_1 = k
                if 'drawer' in k and drawer_1 == 'drawer_1':
                    drawer_1 = k
                if 'location' in k and location_object == 'location_1':
                    location_object = k
                if 'location' in k and location_drawer == 'location_2':
                    location_drawer = k

        # === Plan Execution ===
        try:
            print(f"[Task] Moving to object location: {location_object}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=location_object
            )
            if done:
                print("[Task] Task ended after moving to object location!")
                return

            print(f"[Task] Picking up object: {object_1} at {location_object}")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_1,
                location=location_object
            )
            if done:
                print("[Task] Task ended after picking object!")
                return

            # After pick, need-ready is set. Must go to ready pose.
            print("[Task] Going to ready pose to clear need-ready")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=location_object
            )
            if done:
                print("[Task] Task ended after going to ready pose!")
                return

            print(f"[Task] Moving to drawer location: {location_drawer}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=location_drawer
            )
            if done:
                print("[Task] Task ended after moving to drawer location!")
                return

            print(f"[Task] Placing object: {object_1} into drawer: {drawer_1} at {location_drawer}")
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_1,
                drawer=drawer_1,
                location=location_drawer
            )
            if done:
                print("[Task] Task ended after placing object!")
                return

            # After place, need-ready is set. Must go to ready pose.
            print("[Task] Going to ready pose to clear need-ready")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=location_drawer
            )
            if done:
                print("[Task] Task ended after going to ready pose!")
                return

            # Continue with further steps as needed, always checking for need-ready after actions that set it.

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
