# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use available skills to interact with objects and observe the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and locations to discover missing predicates
        # For demonstration, we will attempt to pick, place, push, pull, sweep, rotate, go, and use gripper on all objects/locations

        # Get all object and location names from positions
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if name.startswith('object') or name.startswith('obj') or name.startswith('handle'):
                object_names.append(name)
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # Assume location is encoded in the name or elsewhere
                pass

        # Fallback: If no explicit locations, use some default ones
        if not location_names:
            location_names = {'drawer', 'bin', 'ready-pose'}

        # Try to execute each skill on each object/location to see if any error or missing predicate is revealed
        for obj in object_names:
            for loc in location_names:
                try:
                    # Try pick
                    if 'execute_pick' in available_skills:
                        print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

                try:
                    # Try place
                    if 'execute_place' in available_skills:
                        print(f"[Exploration] Trying execute_place on {obj} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} at {loc}: {e}")

                try:
                    # Try push
                    if 'execute_push' in available_skills:
                        print(f"[Exploration] Trying execute_push on {obj} at {loc}")
                        obs, reward, done = execute_push(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {obj} at {loc}: {e}")

                try:
                    # Try pull
                    if 'execute_pull' in available_skills:
                        print(f"[Exploration] Trying execute_pull on {obj} at {loc}")
                        obs, reward, done = execute_pull(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {obj} at {loc}: {e}")

                try:
                    # Try sweep
                    if 'execute_sweep' in available_skills:
                        print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

                try:
                    # Try rotate
                    if 'execute_rotate' in available_skills:
                        print(f"[Exploration] Trying execute_rotate on {obj} at {loc}")
                        obs, reward, done = execute_rotate(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj} at {loc}: {e}")

        # Try to move between all pairs of locations
        location_list = list(location_names)
        for i in range(len(location_list)):
            for j in range(len(location_list)):
                if i == j:
                    continue
                from_loc = location_list[i]
                to_loc = location_list[j]
                try:
                    if 'execute_go' in available_skills:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                except Exception as e:
                    print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # Try gripper action if available
        try:
            if 'execute_gripper' in available_skills:
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # After exploration, you would analyze the logs to determine which predicate is missing
        # and update your domain/problem files accordingly for planning.

        # (Optional) If you have an oracle plan, you would execute it here step by step using the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()