# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "execute_go_ready" skill to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Move to an object location
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. (Continue as needed...)

        # For this example, we will use placeholder object and location names.
        # Replace 'object_1', 'location_a', 'drawer_1', etc. with actual names from your environment.

        # Example object and location names (replace with actuals)
        object_1 = None
        location_a = None
        drawer_1 = None
        ready_pose = 'ready-pose'

        # Try to extract at least one object and location from positions
        for k, v in positions.items():
            if object_1 is None:
                object_1 = k
                location_a = v  # Assume v is a location or can be mapped to one
            break  # Only need one for demonstration

        # If you have a way to get drawer names/locations, set drawer_1 accordingly
        # For now, we use a placeholder
        drawer_1 = 'drawer_1'

        # === Plan Execution with need-ready Handling ===
        try:
            # 1. Move to object location
            print(f"[Task] Moving to object location: {location_a}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=location_a
            )
            if done:
                print("[Task] Task ended after move!")
                return

            # 2. Pick the object (sets need-ready)
            print(f"[Task] Picking object: {object_1} at {location_a}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_1,
                location=location_a
            )
            if done:
                print("[Task] Task ended after pick!")
                return

            # 3. Go to ready pose (clears need-ready)
            print(f"[Task] Going to ready pose to clear need-ready")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=location_a,
                to_location=ready_pose
            )
            if done:
                print("[Task] Task ended after go-ready!")
                return

            # 4. Move to drawer location
            print(f"[Task] Moving to drawer: {drawer_1}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_1
            )
            if done:
                print("[Task] Task ended after move to drawer!")
                return

            # 5. Place the object into the drawer (sets need-ready)
            print(f"[Task] Placing object: {object_1} into drawer: {drawer_1}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_1,
                drawer_name=drawer_1,
                location=drawer_1
            )
            if done:
                print("[Task] Task ended after place!")
                return

            # 6. Go to ready pose (clears need-ready)
            print(f"[Task] Going to ready pose to clear need-ready after place")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=drawer_1,
                to_location=ready_pose
            )
            if done:
                print("[Task] Task ended after go-ready (post-place)!")
                return

            # Continue with further plan steps as needed...

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

        # === Exploration Knowledge Usage ===
        # If you need to further explore for missing predicates, you could use the exploration actions
        # (e.g., execute_go_identify, execute_pick_weight, etc.) if they are available in skill_code.
        # For now, we focus on handling (need-ready) as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
