# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block skill execution.
        # We will attempt to execute a skill, and if it fails due to (need-ready), we will call execute_go_ready.

        # For demonstration, let's try to pick an object from the floor if available.
        # We'll use only the provided skills and available objects/positions.

        # Find an object on the floor (excluding handles)
        object_on_floor = None
        object_pos = None
        for obj_name, pos_info in positions.items():
            # Assume pos_info contains at least 'location' and 'is_handle'
            if pos_info.get('on_floor', False) and not pos_info.get('is_handle', False):
                object_on_floor = obj_name
                object_pos = pos_info.get('location', None)
                break

        # Find a location for the robot to be at (use object's location if available)
        robot_location = None
        for obj_name, pos_info in positions.items():
            if pos_info.get('robot', False):
                robot_location = pos_info.get('location', None)
                break
        if robot_location is None and object_pos is not None:
            robot_location = object_pos

        # Main plan: Try to pick the object, handle (need-ready) if encountered
        if object_on_floor is not None and object_pos is not None:
            print(f"[Exploration] Attempting to pick object '{object_on_floor}' at location {object_pos}")
            try:
                # Try to move to the object's location if not already there
                if robot_location != object_pos:
                    print(f"[Exploration] Moving robot from {robot_location} to {object_pos}")
                    obs, reward, done = execute_go(env, task, robot_location, object_pos)
                    robot_location = object_pos

                # Try to pick the object
                obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
                print(f"[Exploration] Picked object '{object_on_floor}' successfully.")
            except Exception as e:
                # If the error is due to (need-ready), call execute_go_ready and retry
                error_msg = str(e)
                if "need-ready" in error_msg or "NeedReady" in error_msg:
                    print("[Exploration] Detected (need-ready) predicate. Executing execute_go_ready to reset.")
                    try:
                        obs, reward, done = execute_go_ready(env, task, robot_location)
                        print("[Exploration] Executed execute_go_ready. Retrying pick.")
                        # Retry pick after resetting
                        obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
                        print(f"[Exploration] Picked object '{object_on_floor}' after resetting need-ready.")
                    except Exception as e2:
                        print(f"[Exploration] Failed to pick object after resetting need-ready: {e2}")
                else:
                    print(f"[Exploration] Failed to pick object '{object_on_floor}': {e}")

        else:
            print("[Exploration] No suitable object found on the floor to pick.")

        # === Continue with further plan steps as needed ===
        # For a full task, you would continue to use the available skills (execute_place, execute_push, etc.)
        # and always check for (need-ready) feedback, calling execute_go_ready as needed.

        # Example: Place the object into a drawer if possible
        # (This is a placeholder; actual logic would depend on your environment and plan)
        # drawer_name, drawer_location = ... (find from positions)
        # try:
        #     obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_location)
        # except Exception as e:
        #     if "need-ready" in str(e):
        #         obs, reward, done = execute_go_ready(env, task, robot_location)
        #         obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_location)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
