# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (not (need-ready)) ===
        # The feedback indicates that the predicate (not (need-ready)) is missing or not being handled.
        # According to the domain, many actions require (not (need-ready)) as a precondition.
        # Therefore, we must ensure that the robot is not in the 'need-ready' state before executing such actions.
        # The only way to clear 'need-ready' is to execute the 'execute_go_ready' skill, which removes (need-ready).
        # We will attempt to execute a sequence of actions, and if an action fails due to 'need-ready', we will call 'execute_go_ready'.

        # For demonstration, let's attempt to pick and place an object, handling 'need-ready' as needed.
        # We'll use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this exploration, let's try to pick up the first object on the floor and place it somewhere (e.g., in a bin or drawer).
        # We'll use the positions dictionary to find an object and its location.

        # Find an object on the floor
        object_to_pick = None
        object_pos = None
        for obj_name, pos_info in positions.items():
            # Assume pos_info contains keys like 'on_floor', 'location', etc.
            if pos_info.get('on_floor', False):
                object_to_pick = obj_name
                object_pos = pos_info.get('location', None)
                break

        if object_to_pick is None or object_pos is None:
            print("[Exploration] No object found on the floor to pick.")
        else:
            print(f"[Exploration] Attempting to pick object '{object_to_pick}' at location '{object_pos}'.")

            # Assume the robot starts at some location; get robot's current location
            robot_location = None
            for obj_name, pos_info in positions.items():
                if obj_name == 'robot':
                    robot_location = pos_info.get('location', None)
                    break

            if robot_location is None:
                print("[Exploration] Robot location unknown, cannot proceed.")
            else:
                # Move to the object's location if not already there
                if robot_location != object_pos:
                    try:
                        print(f"[Exploration] Moving robot from '{robot_location}' to '{object_pos}'.")
                        obs, reward, done = execute_go(env, task, robot_location, object_pos)
                        robot_location = object_pos
                    except Exception as e:
                        print(f"[Exploration] Error during execute_go: {e}")

                # Try to pick the object, handling 'need-ready' if necessary
                picked = False
                for attempt in range(2):
                    try:
                        print(f"[Exploration] Attempt {attempt+1}: Trying to pick '{object_to_pick}' at '{object_pos}'.")
                        obs, reward, done = execute_pick(env, task, object_to_pick, object_pos)
                        picked = True
                        print(f"[Exploration] Successfully picked '{object_to_pick}'.")
                        break
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed: {e}")
                        print("[Exploration] Attempting to clear 'need-ready' by calling execute_go_ready.")
                        try:
                            obs, reward, done = execute_go_ready(env, task, object_pos)
                        except Exception as e2:
                            print(f"[Exploration] execute_go_ready failed: {e2}")
                            break

                if not picked:
                    print(f"[Exploration] Could not pick '{object_to_pick}' after handling 'need-ready'.")
                else:
                    # Try to place the object into a bin or drawer (if available)
                    # For demonstration, look for a 'bin' or 'drawer' in positions
                    place_target = None
                    place_location = None
                    for obj_name, pos_info in positions.items():
                        if obj_name.startswith('bin') or obj_name.startswith('drawer'):
                            place_target = obj_name
                            place_location = pos_info.get('location', None)
                            break

                    if place_target is None or place_location is None:
                        print("[Exploration] No bin or drawer found to place the object.")
                    else:
                        # Move to the place location if needed
                        if robot_location != place_location:
                            try:
                                print(f"[Exploration] Moving robot from '{robot_location}' to '{place_location}'.")
                                obs, reward, done = execute_go(env, task, robot_location, place_location)
                                robot_location = place_location
                            except Exception as e:
                                print(f"[Exploration] Error during execute_go: {e}")

                        # Try to place the object, handling 'need-ready' if necessary
                        placed = False
                        for attempt in range(2):
                            try:
                                print(f"[Exploration] Attempt {attempt+1}: Trying to place '{object_to_pick}' into '{place_target}' at '{place_location}'.")
                                obs, reward, done = execute_place(env, task, object_to_pick, place_target, place_location)
                                placed = True
                                print(f"[Exploration] Successfully placed '{object_to_pick}' into '{place_target}'.")
                                break
                            except Exception as e:
                                print(f"[Exploration] execute_place failed: {e}")
                                print("[Exploration] Attempting to clear 'need-ready' by calling execute_go_ready.")
                                try:
                                    obs, reward, done = execute_go_ready(env, task, place_location)
                                except Exception as e2:
                                    print(f"[Exploration] execute_go_ready failed: {e2}")
                                    break

                        if not placed:
                            print(f"[Exploration] Could not place '{object_to_pick}' after handling 'need-ready'.")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
