# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Pick an object from the floor (execute_pick)
        # 2. Place it into a drawer (execute_place)
        # 3. If (need-ready) is set, call execute_go_ready
        # 4. Move to another location (execute_go)
        # 5. Repeat as needed

        # For this example, we will:
        # - Find an object on the floor
        # - Find a drawer and its location
        # - Pick the object, place it in the drawer, and handle (need-ready)

        # --- Find an object on the floor ---
        object_on_floor = None
        object_on_floor_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_on_floor = obj_name
                object_on_floor_pos = obj_info['position']
                break

        if object_on_floor is None:
            print("[Exploration] No object found on the floor. Exiting.")
            return

        # --- Find a drawer and its location ---
        drawer_name = None
        drawer_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') == 'drawer':
                drawer_name = obj_name
                drawer_pos = obj_info['position']
                break

        if drawer_name is None:
            print("[Exploration] No drawer found in the environment. Exiting.")
            return

        # --- Get robot's current location ---
        robot_location = None
        for obj_name, obj_info in positions.items():
            if obj_name == 'robot':
                robot_location = obj_info['position']
                break

        if robot_location is None:
            # Fallback: use a default location if not found
            robot_location = [0, 0, 0]

        # === Plan Execution ===

        # 1. Move to the object's location if not already there
        try:
            if not np.allclose(robot_location, object_on_floor_pos, atol=0.05):
                print(f"[Task] Moving to object {object_on_floor} at {object_on_floor_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=object_on_floor_pos
                )
                robot_location = object_on_floor_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # 2. Pick the object from the floor
        try:
            print(f"[Task] Picking object {object_on_floor} at {object_on_floor_pos}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_on_floor,
                location=object_on_floor_pos
            )
            # After execute_pick, (need-ready) is set
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 3. If (need-ready) is set, call execute_go_ready
        try:
            # In this skeleton, we assume (need-ready) is always set after pick
            print("[Task] Executing go_ready to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_on_floor_pos
            )
            # Robot is now at 'ready-pose'
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to execute go_ready: {e}")
            return

        # 4. Move to the drawer location
        try:
            print(f"[Task] Moving to drawer {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_pos
            )
            robot_location = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # 5. Place the object into the drawer
        try:
            print(f"[Task] Placing object {object_on_floor} into drawer {drawer_name}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_on_floor,
                drawer_name=drawer_name,
                location=drawer_pos
            )
            # After execute_place, (need-ready) is set
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")
            return

        # 6. If (need-ready) is set, call execute_go_ready again
        try:
            print("[Task] Executing go_ready to clear (need-ready) after place")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos
            )
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to execute go_ready after place: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
