# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (not (need-ready)) ===
        # The feedback indicates that the predicate (not (need-ready)) is missing or not being handled.
        # We will explore the environment using available skills to determine when (need-ready) is set/unset.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to execute a sequence of skills and observe the effect on the 'need-ready' predicate.
        # Since we do not have direct access to the predicate, we will infer its effect by the success/failure of actions that require (not (need-ready)) in their preconditions.

        # For demonstration, we will:
        # 1. Try to pick an object (should set need-ready).
        # 2. Try to move (should fail if need-ready is True).
        # 3. Execute go_ready to reset need-ready.
        # 4. Try to move again (should succeed if need-ready is False).

        # --- Step 1: Find an object on the floor to pick ---
        object_to_pick = None
        pick_position = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_to_pick = obj_name
                pick_position = obj_info.get('position', None)
                break

        if object_to_pick is None or pick_position is None:
            print("[Exploration] No suitable object found on the floor to pick.")
        else:
            print(f"[Exploration] Attempting to pick object '{object_to_pick}' at position {pick_position}.")

            try:
                # Try to pick the object (should set need-ready)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_to_pick,
                    target_pos=pick_position,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Pick executed. (need-ready) should now be True.")
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

            # --- Step 2: Try to move (should fail if need-ready is True) ---
            # Find two locations to move between
            locations = [k for k, v in positions.items() if v.get('type', '') == 'location']
            if len(locations) >= 2:
                from_loc = locations[0]
                to_loc = locations[1]
                print(f"[Exploration] Attempting to move from '{from_loc}' to '{to_loc}' (should fail if need-ready is True).")
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=from_loc,
                        to_location=to_loc,
                        max_steps=100,
                        timeout=10.0
                    )
                    print("[Exploration] Move executed. (need-ready) was likely False.")
                except Exception as e:
                    print(f"[Exploration] Move failed as expected due to (need-ready): {e}")
            else:
                print("[Exploration] Not enough locations to test move.")

            # --- Step 3: Execute go_ready to reset need-ready ---
            print("[Exploration] Executing go_ready to reset (need-ready).")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=from_loc,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Exploration] go_ready executed. (need-ready) should now be False.")
            except Exception as e:
                print(f"[Exploration] Exception during go_ready: {e}")

            # --- Step 4: Try to move again (should succeed if need-ready is False) ---
            print(f"[Exploration] Attempting to move from '{from_loc}' to '{to_loc}' again (should succeed if need-ready is False).")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Exploration] Move executed successfully after resetting (need-ready).")
            except Exception as e:
                print(f"[Exploration] Move failed unexpectedly: {e}")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()