# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and some require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it before proceeding.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick_object)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Move to a drawer (execute_go)
        # 4. Place the object into the drawer (execute_place_to_drawer)
        # 5. Go to ready pose (execute_go_ready) again if needed

        # For this example, we will use placeholder names for objects, locations, and drawers.
        # Replace these with actual names from your environment as needed.

        # --- Example object and location selection ---
        # Find an object on the floor (not a handle)
        object_name = None
        object_pos = None
        for name, pos in positions.items():
            if 'handle' not in name and 'drawer' not in name and 'bin' not in name:
                object_name = name
                object_pos = pos
                break

        # Find a drawer and its location
        drawer_name = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_name = name
                drawer_pos = pos
                break

        # Find a location for the robot to start from
        robot_location = None
        for name, pos in positions.items():
            if 'ready' in name or 'start' in name or 'pose' in name:
                robot_location = name
                break
        if robot_location is None:
            robot_location = 'ready-pose'  # fallback

        # For demonstration, we use string names; adapt as needed for your environment

        # --- Plan Execution with need-ready handling ---
        try:
            print(f"[Task] Picking up object: {object_name} at {robot_location}")
            obs, reward, done = execute_pick_object(
                env,
                task,
                obj=object_name,
                location=robot_location
            )
            # After execute_pick_object, (need-ready) is set. Must call execute_go_ready.
            print("[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location
            )

            # Move to drawer location (if not already there)
            if drawer_pos is not None and robot_location != drawer_name:
                print(f"[Task] Moving to drawer: {drawer_name}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_name
                )
                robot_location = drawer_name

            # Place object into drawer
            print(f"[Task] Placing object {object_name} into drawer {drawer_name}")
            obs, reward, done = execute_place_to_drawer(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=robot_location
            )
            # After execute_place_to_drawer, (need-ready) is set. Must call execute_go_ready.
            print("[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location
            )

            print("[Task] Task completed successfully.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
