# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that blocks some actions.
        # We must ensure that after actions that set (need-ready), we call the appropriate skill to clear it.
        # In the domain, execute_go_ready clears (need-ready).
        # So, after any skill that sets (need-ready), we must call execute_go_ready.

        # For demonstration, let's assume a generic plan:
        # 1. Pick up an object from the floor (execute_pick_object)
        # 2. Place it into a drawer (execute_place_to_drawer)
        # 3. If (need-ready) is set after pick/place, call execute_go_ready to clear it
        # 4. Move to another location (execute_go)
        # 5. Push/pull drawer as needed

        # For this example, let's use placeholder names for objects, drawers, locations
        # You should replace these with actual names from your environment/observation

        # Example object and location names (replace with actuals)
        object_name = None
        drawer_name = None
        handle_name = None
        robot_location = None
        target_location = None

        # Try to extract at least one object, drawer, handle, and location from positions
        for name in positions:
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
            elif 'handle' in name and handle_name is None:
                handle_name = name
            elif 'object' in name and object_name is None:
                object_name = name
            elif 'bin' in name and target_location is None:
                target_location = name
            elif robot_location is None:
                robot_location = name

        # Fallbacks if not found
        if object_name is None:
            object_name = 'object_1'
        if drawer_name is None:
            drawer_name = 'drawer_1'
        if handle_name is None:
            handle_name = 'handle_1'
        if robot_location is None:
            robot_location = 'ready-pose'
        if target_location is None:
            target_location = 'bin'

        # === Example Task Plan with Exploration for (need-ready) ===

        try:
            print(f"[Task] Attempting to pick up {object_name} at {robot_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_name,
                location=robot_location
            )
            # After execute_pick, (need-ready) is set, so we must call execute_go_ready
            print("[Task] (need-ready) set, calling execute_go_ready to clear it")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location
            )

            print(f"[Task] Moving to {drawer_name} location")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_name
            )

            print(f"[Task] Placing {object_name} into {drawer_name}")
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=drawer_name
            )
            # After execute_place, (need-ready) is set, so call execute_go_ready again
            print("[Task] (need-ready) set, calling execute_go_ready to clear it")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_name
            )

            print(f"[Task] Task completed for {object_name} into {drawer_name}")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

        # === Exploration Knowledge: If you need to identify, weigh, or check lock, use the corresponding skills ===
        # For example, if you need to identify an object at a location:
        # obs, reward, done = execute_go_identify(env, task, robot, from_location, to_location)
        # If you need to check weight: obs, reward, done = execute_pick_weight(env, task, robot, obj, location)
        # If you need to check lock: obs, reward, done = execute_pull(env, task, robot, obj, location)
        # (These are only available if provided in skill_code.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
