# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (drawer-unlocked drawer_top) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer_top) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this, we will attempt to pull the drawer using execute_pull.
        # If the drawer is locked, the action will fail or not have the expected effect.
        # If the drawer is unlocked, the action will succeed and the drawer will open.

        # --- Identify relevant objects and locations ---
        # These names are placeholders; replace with actual names from your environment if needed.
        drawer_name = 'drawer_top'
        handle_name = 'handle_top'
        robot_location = 'ready-pose'
        drawer_location = 'drawer_top_front'
        # The handle is assumed to be attached to the drawer and accessible at drawer_location.

        # Step 1: Move robot to the drawer location
        try:
            print("[Exploration] Moving robot to drawer location:", drawer_location)
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location
            )
            robot_location = drawer_location
        except Exception as e:
            print("[Exploration] Failed to move to drawer location:", e)
            return

        # Step 2: Pick the handle if not already holding it
        try:
            print("[Exploration] Attempting to pick handle:", handle_name)
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_location
            )
        except Exception as e:
            print("[Exploration] Failed to pick handle:", e)
            return

        # Step 3: Attempt to pull the drawer open
        try:
            print("[Exploration] Attempting to pull the drawer to check lock state.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            # If this succeeds, the drawer is unlocked.
            print("[Exploration] Drawer pull succeeded: drawer is unlocked.")
            drawer_unlocked = True
        except Exception as e:
            print("[Exploration] Drawer pull failed, likely locked:", e)
            drawer_unlocked = False

        # Step 4: If the drawer is locked, report missing predicate
        if not drawer_unlocked:
            print("[Exploration] Missing predicate detected: (drawer-unlocked {})".format(drawer_name))
            # In a real system, you might log this or trigger a learning update.

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan if drawer is unlocked ===
        if drawer_unlocked:
            # Example: Place an object into the drawer
            object_to_place = 'object_1'
            try:
                print("[Task] Picking object from floor:", object_to_place)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_to_place,
                    location=robot_location
                )
            except Exception as e:
                print("[Task] Failed to pick object:", e)
                return

            try:
                print("[Task] Placing object into drawer:", drawer_name)
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=object_to_place,
                    drawer_name=drawer_name,
                    location=drawer_location
                )
            except Exception as e:
                print("[Task] Failed to place object into drawer:", e)
                return

            print("[Task] Oracle plan executed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
