# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call the skill that clears it.
        # The action that clears (need-ready) is execute_go_ready.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (if available) using execute_pick or execute_pick_object.
        # 2. If the action sets (need-ready), immediately call execute_go_ready to clear it.
        # 3. Continue with other actions as needed.

        # === Example: Find an object on the floor and pick it up ===
        # We'll look for an object that is on the floor and not a handle.
        # For this, we assume positions contains keys like 'objects', 'drawers', 'locations', etc.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Try to find a pickable object and a location
        object_found = None
        object_pos = None
        robot_location = None

        # Try to extract a pickable object and its location
        try:
            # Example: positions['objects'] = {'obj1': {'type': 'object', 'on_floor': True, 'is_handle': False, 'location': 'loc1'}, ...}
            for obj_name, obj_info in positions.get('objects', {}).items():
                if obj_info.get('on_floor', False) and not obj_info.get('is_handle', False):
                    object_found = obj_name
                    object_pos = obj_info.get('location', None)
                    break
        except Exception as e:
            print("[Exploration] Could not find a pickable object:", e)

        # Try to get robot's current location
        try:
            robot_location = positions.get('robot', {}).get('location', None)
        except Exception as e:
            print("[Exploration] Could not get robot location:", e)

        # If we have both, try to pick the object
        if object_found and object_pos and robot_location:
            print(f"[Exploration] Attempting to pick object '{object_found}' at location '{object_pos}'.")

            try:
                # Move robot to the object's location if not already there
                if robot_location != object_pos:
                    print(f"[Exploration] Moving robot from '{robot_location}' to '{object_pos}'.")
                    obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=object_pos)
                    robot_location = object_pos

                # Try to pick the object
                obs, reward, done = execute_pick(env, task, object_name=object_found, location=object_pos)
                print(f"[Exploration] Picked object '{object_found}'.")
            except Exception as e:
                print(f"[Exploration] Failed to pick object '{object_found}':", e)
                obs, reward, done = None, 0, False

            # After picking, (need-ready) is likely set. Try to clear it.
            try:
                print("[Exploration] Executing execute_go_ready to clear (need-ready).")
                obs, reward, done = execute_go_ready(env, task, from_location=object_pos)
                print("[Exploration] (need-ready) cleared.")
            except Exception as e:
                print("[Exploration] Failed to execute execute_go_ready:", e)
        else:
            print("[Exploration] No suitable object found for picking.")

        # === Continue with further actions as needed ===
        # For a full plan, you would continue to use the available skills in the correct order,
        # always checking if (need-ready) is set after actions that set it, and calling execute_go_ready as needed.

        # Example: Place the object into a drawer or bin if desired, always clearing (need-ready) after such actions.

        # TODO: Implement further plan steps as required by your specific task.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
