# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick_object)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Place the object into a drawer (execute_place_to_drawer)
        # 4. Go to ready pose (execute_go_ready) again

        # For this example, we assume the following:
        # - There is at least one object on the floor (not a handle)
        # - There is at least one drawer and one location
        # - The robot starts at a known location

        # Find a non-handle object on the floor
        object_name = None
        object_pos = None
        for name, info in positions.items():
            if info.get('on_floor', False) and not info.get('is_handle', False):
                object_name = name
                object_pos = info['position']
                object_location = info['location']
                break

        if object_name is None:
            print("[Exploration] No suitable object found on the floor to pick.")
            return

        # Find a drawer and its location
        drawer_name = None
        drawer_location = None
        for name, info in positions.items():
            if info.get('is_drawer', False):
                drawer_name = name
                drawer_location = info['location']
                break

        if drawer_name is None:
            print("[Exploration] No drawer found in the environment.")
            return

        # Find robot's current location
        robot_location = None
        for name, info in positions.items():
            if name == 'robot':
                robot_location = info['location']
                break

        if robot_location is None:
            print("[Exploration] Robot location not found.")
            return

        # 1. Move to the object's location if not already there
        if robot_location != object_location:
            print(f"[Task] Moving robot from {robot_location} to {object_location}")
            try:
                obs, reward, done = execute_go(env, task, robot_location, object_location)
                robot_location = object_location
            except Exception as e:
                print(f"[Error] Failed to move robot: {e}")
                return

        # 2. Pick up the object from the floor
        print(f"[Task] Picking up object {object_name} at {object_location}")
        try:
            obs, reward, done = execute_pick(env, task, object_name, object_location)
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 3. After picking, (need-ready) is set. Must call execute_go_ready to clear it.
        print("[Task] Going to ready pose to clear (need-ready)")
        try:
            obs, reward, done = execute_go_ready(env, task, object_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # 4. Move to the drawer's location if not already there
        if robot_location != drawer_location:
            print(f"[Task] Moving robot from {robot_location} to {drawer_location}")
            try:
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Error] Failed to move robot: {e}")
                return

        # 5. Place the object into the drawer
        print(f"[Task] Placing object {object_name} into drawer {drawer_name} at {drawer_location}")
        try:
            obs, reward, done = execute_place(env, task, object_name, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")
            return

        # 6. After placing, (need-ready) is set again. Call execute_go_ready.
        print("[Task] Going to ready pose to clear (need-ready) after placing")
        try:
            obs, reward, done = execute_go_ready(env, task, drawer_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
