# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (not (need-ready)) is missing in the state, which blocks skill execution.
        # We need to explore the environment to clear the (need-ready) predicate if present.

        # The plan: If (need-ready) is True, we must execute the 'execute_go_ready' skill
        # to clear it. Only then can we proceed with the main task plan.

        # We'll try to execute a dummy action (e.g., pick or move). If it fails due to (need-ready),
        # we execute 'execute_go_ready' and retry.

        # For demonstration, let's try to pick an object if available, else move.

        # Find a pickable object on the floor
        pickable_obj = None
        pickable_obj_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False) and not obj_info.get('is_handle', False):
                pickable_obj = obj_name
                pickable_obj_pos = obj_info['position']
                break

        # Find robot's current location
        robot_location = None
        for obj_name, obj_info in positions.items():
            if obj_name == 'robot':
                robot_location = obj_info.get('location', None)
                break
        if robot_location is None:
            # Fallback: try to get from obs or default
            robot_location = 'ready-pose'

        # Try to execute a skill, handle (need-ready) if present
        def try_skill_with_need_ready_clear(skill_func, *args, **kwargs):
            try:
                obs, reward, done = skill_func(env, task, *args, **kwargs)
                return obs, reward, done
            except Exception as e:
                # Check if error is due to (need-ready)
                if 'need-ready' in str(e) or 'NeedReady' in str(e):
                    print("[Exploration] (need-ready) detected, executing execute_go_ready to clear.")
                    try:
                        # Try to execute execute_go_ready
                        obs, reward, done = execute_go_ready(env, task, from_location=robot_location)
                        print("[Exploration] (need-ready) cleared.")
                    except Exception as e2:
                        print("[Exploration] Failed to clear (need-ready):", e2)
                        return None, 0, True
                    # Retry the original skill
                    try:
                        obs, reward, done = skill_func(env, task, *args, **kwargs)
                        return obs, reward, done
                    except Exception as e3:
                        print("[Exploration] Skill failed again after clearing (need-ready):", e3)
                        return None, 0, True
                else:
                    print("[Exploration] Skill failed for another reason:", e)
                    return None, 0, True

        # === Example Task Plan ===
        # 1. If pickable object exists, try to pick it
        if pickable_obj is not None:
            print(f"[Task] Attempting to pick object '{pickable_obj}' at {pickable_obj_pos}")
            obs, reward, done = try_skill_with_need_ready_clear(
                execute_pick, 
                obj_name=pickable_obj, 
                location=robot_location
            )
            if done:
                print("[Task] Task ended after picking object!")
                return
        else:
            # If no pickable object, try to move somewhere else
            # Find a different location to move to
            all_locations = set()
            for obj_name, obj_info in positions.items():
                loc = obj_info.get('location', None)
                if loc is not None:
                    all_locations.add(loc)
            all_locations = list(all_locations)
            if len(all_locations) > 1:
                to_location = [loc for loc in all_locations if loc != robot_location][0]
                print(f"[Task] Attempting to move from {robot_location} to {to_location}")
                obs, reward, done = try_skill_with_need_ready_clear(
                    execute_go,
                    from_location=robot_location,
                    to_location=to_location
                )
                if done:
                    print("[Task] Task ended after move!")
                    return
            else:
                print("[Task] No alternative location to move to. Exploration ends.")

        # === Continue with further plan steps as needed ===
        # (You can add more plan steps here, using try_skill_with_need_ready_clear to ensure (need-ready) is handled.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
