# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to trigger any observable state changes or errors.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill in a safe, exploratory way to see if any predicate is required/missing
        # For demonstration, we will attempt to pick and pull objects, and move between locations

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # Fallback: If no explicit location names, try to infer from object positions
        if not location_names:
            # Assume at least one location exists, e.g., 'table', 'drawer', 'bin', etc.
            possible_locations = ['table', 'drawer', 'bin', 'ready-pose']
            for loc in possible_locations:
                if loc in positions:
                    location_names.append(loc)

        # Try to move to each location (explore with execute_go)
        current_location = None
        for loc in location_names:
            try:
                if current_location is not None and loc != current_location:
                    print(f"[Exploration] Moving from {current_location} to {loc} using execute_go")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_location,
                        to_location=loc
                    )
                    current_location = loc
                elif current_location is None:
                    # Move to the first location from wherever we are
                    print(f"[Exploration] Moving to {loc} using execute_go")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=loc,
                        to_location=loc
                    )
                    current_location = loc
            except Exception as e:
                print(f"[Exploration] execute_go failed for location {loc}: {e}")

        # Try to pick each object (explore with execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc} using execute_pick")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj,
                        location_name=loc
                    )
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # Try to pull each object (explore with execute_pull)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pull {obj} at {loc} using execute_pull")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        object_name=obj,
                        location_name=loc
                    )
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {obj} at {loc}: {e}")

        # Try to place each object (explore with execute_place)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to place {obj} at {loc} using execute_place")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj,
                        location_name=loc
                    )
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} at {loc}: {e}")

        # Try to push at each location (explore with execute_push)
        for loc in location_names:
            try:
                print(f"[Exploration] Trying to push at {loc} using execute_push")
                obs, reward, done = execute_push(
                    env,
                    task,
                    location_name=loc
                )
            except Exception as e:
                print(f"[Exploration] execute_push failed at {loc}: {e}")

        # Try to sweep each object (explore with execute_sweep)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc} using execute_sweep")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        object_name=obj,
                        location_name=loc
                    )
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # Try to rotate each object (explore with execute_rotate)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to rotate {obj} at {loc} using execute_rotate")
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        object_name=obj,
                        location_name=loc
                    )
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj} at {loc}: {e}")

        # Try to use gripper (explore with execute_gripper)
        try:
            print(f"[Exploration] Trying to use gripper with execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from exploration and feedback,
        # update the domain/problem or inform the user to add the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()