# run_skeleton_task.py (Completed with Exploration Logic and Missing Predicate Diagnosis)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # At this point positions is expected to be a dict mapping object names to their position info.
        # For predicates check and exploration, let's log what objects exist:
        print("[Object Position Debug] Object names available:", list(positions.keys()))

        # ================== EXPLORATION PHASE ==================

        # Problem setup: Feedback indicates (holding drawer_top_handle) is missing. This means the state
        # does not reflect the robot holding the handle, or preconditions/effects regarding holding 
        # (specifically for "handle" objects) are not being registered correctly.

        # To determine if the 'holding' predicate (specifically for a handle object) is missing,
        # we'll attempt to perform actions involving picking up a handle, then inspect the state.
        # If 'holding <handle>' is not set after a successful pick, we have found the missing predicate.

        # Infer handle name:
        handle_candidates = [name for name in positions if 'handle' in name]
        if not handle_candidates:
            print("[Exploration] No handle-like objects found in positions. Exploration cannot proceed.")
        else:
            drawer_handle = handle_candidates[0]  # e.g. 'drawer_top_handle'
            print(f"[Exploration] Using handle candidate: {drawer_handle}")

            # Next, get a plausible location for the handle
            handle_pos = positions[drawer_handle]
            # For location, perhaps simulation provides a mapping. We'll just use the position value as 'handle_location'
            handle_location = 'handle_location'
            # In actual system, map to a symbolic location if needed.

            # EXECUTION ATTEMPT: Try to pick the handle with the designated skill.
            exploration_success = False
            try:
                print(f"[Exploration] Attempting to pick handle: {drawer_handle}")
                # Try using the relevant skill. Name in available skills: 'execute_pick'
                # Skill signature assumed: execute_pick(env, task, obj_name, location, **kwargs)
                _obs, _reward, _done, info = execute_pick(
                    env, task,
                    obj_name=drawer_handle,
                    location=handle_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_success = True
                print("[Exploration] Handle pick action executed.")

            except Exception as e:
                print(f"[Exploration] Exception during handle pick: {e}")

            # (OPTIONAL) Check task state for 'holding' predicate, if possible.
            # In reality, you might query task.pddl_state() or similar.
            # Here, we'll print/show the info returned (if skill provides state)
            if exploration_success:
                if hasattr(task, 'get_current_predicates'):
                    # Try to obtain current predicates (debug only)
                    predicates = task.get_current_predicates()
                    print("[Exploration] Current Predicates after handle pick:", predicates)
                    has_holding_handle = any(
                        p for p in predicates if p[0] == 'holding' and drawer_handle in p
                    )
                    if not has_holding_handle:
                        print(">>> [Missing Predicate Identified] : 'holding' for handle is missing.")
                else:
                    print("[Exploration] Cannot directly check predicates; inspect via environment logs.")

        # ================== PLAN/MAIN TASK PHASE ==================

        # After exploration, you would normally run your actual oracle plan.
        # The following example uses only available predefined skills.
        #
        # Replace the steps below with the true oracle plan steps, filling in parameters as required.

        # This section is generalized; adapt as needed:

        # Example manipulation plan:
        print("[Task Phase] Executing oracle plan using available skills...")

        # 1. Move robot to location of handle object
        try:
            # We need to know current robot location; let's assume there's a function or we know the initial pose.
            # Here we use 'start_pose' as a placeholder. In your setup, replace accordingly.
            current_location = 'start_pose'
            to_location = handle_location
            print(f"[Task] Moving from {current_location} to {to_location}.")
            _obs, _reward, _done, info = execute_go(
                env, task,
                from_location=current_location,
                to_location=to_location,
                trajectory_type='default',
                max_steps=60,
                threshold=0.01,
                timeout=5.0
            )
            print(f"[Task] Arrived at {to_location}.")

        except Exception as e:
            print(f"[Task] Exception during execute_go: {e}")

        # 2. Pick up the handle if not already done
        try:
            print(f"[Task] Trying to pick handle {drawer_handle} at {to_location}.")
            _obs, _reward, _done, info = execute_pick(
                env, task,
                obj_name=drawer_handle,
                location=to_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print(f"[Task] Picked up the handle: {drawer_handle}.")

        except Exception as e:
            print(f"[Task] Exception while picking handle: {e}")
        
        # 3. [Optional] If the plan requires, perform "pull" using the handle
        try:
            # Suppose we know the drawer name from the handle, e.g., 'top_drawer'
            possible_drawers = [name for name in positions if 'drawer' in name and name != drawer_handle]
            if not possible_drawers:
                print("[Task] No drawer objects found for pull.")
            else:
                drawer_name = possible_drawers[0]
                print(f"[Task] Attempting to pull open drawer {drawer_name} using handle {drawer_handle}.")
                # Skill: execute_pull(env, task, drawer_name, handle_name, location, **kwargs)
                _obs, _reward, _done, info = execute_pull(
                    env, task,
                    drawer_name=drawer_name,
                    handle_name=drawer_handle,
                    location=to_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Pulled open drawer {drawer_name}.")

        except Exception as e:
            print(f"[Task] Exception while pulling drawer: {e}")

        # 4. [Insert additional oracle-step skills as needed; only use provided skill set]

        print("===== End of Oracle Plan Phase =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
