# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll attempt to use available skills to explore the environment and infer which predicate is missing.
        # Since we cannot define new skills, we use only the available ones.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to move the robot to all known locations to see if any action fails due to a missing predicate.
        # 2. Try to pick, place, push, pull, sweep, rotate, and use gripper on objects to see if any action fails.
        # 3. Log any exceptions or failures that may indicate a missing predicate.

        # For demonstration, we attempt to move and interact with all objects and drawers.
        # This is a generic exploration loop.

        # Get all objects and locations from the environment
        object_names = [name for name in positions.keys() if not name.startswith('drawer') and not name.startswith('bin')]
        drawer_names = [name for name in positions.keys() if name.startswith('drawer')]
        bin_names = [name for name in positions.keys() if name.startswith('bin')]
        location_names = list(set([v for v in positions.values()]))

        # Try to move to each location
        current_location = None
        try:
            # Try to get the robot's current location from the observation or positions
            if 'robot' in positions:
                current_location = positions['robot']
            elif 'robot-at' in descriptions:
                current_location = descriptions['robot-at']
            else:
                current_location = None
        except Exception:
            current_location = None

        # If we can't determine the current location, pick the first location as a starting point
        if current_location is None and len(location_names) > 0:
            current_location = location_names[0]

        # Exploration: Try to move to all locations
        for loc in location_names:
            if current_location != loc:
                try:
                    print(f"[Exploration] Trying to move from {current_location} to {loc}")
                    obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                    current_location = loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")

        # Exploration: Try to pick all objects from the floor
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick object {obj} at {current_location}")
                obs, reward, done = execute_pick(env, task, object_name=obj, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # Exploration: Try to pull all drawers with all handles (if handles are objects)
        for drawer in drawer_names:
            for obj in object_names:
                try:
                    print(f"[Exploration] Trying to pull drawer {drawer} with object {obj} at {current_location}")
                    obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=obj, location=current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pull {drawer} with {obj}: {e}")

        # Exploration: Try to push all drawers
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to push drawer {drawer} at {current_location}")
                obs, reward, done = execute_push(env, task, drawer_name=drawer, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to push {drawer}: {e}")

        # Exploration: Try to place all held objects into all drawers and bins
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place object {obj} into drawer {drawer} at {current_location}")
                    obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer, location=current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} into {drawer}: {e}")
            for bin_name in bin_names:
                try:
                    print(f"[Exploration] Trying to place object {obj} into bin {bin_name} at {current_location}")
                    obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=bin_name, location=current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} into {bin_name}: {e}")

        # Exploration: Try to sweep and rotate all objects
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep object {obj} at {current_location}")
                obs, reward, done = execute_sweep(env, task, object_name=obj, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")
            try:
                print(f"[Exploration] Trying to rotate object {obj} at {current_location}")
                obs, reward, done = execute_rotate(env, task, object_name=obj, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to rotate {obj}: {e}")

        # Exploration: Try to use gripper
        try:
            print(f"[Exploration] Trying to use gripper at {current_location}")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for any failed actions that may indicate missing predicates.")

        # === End of Exploration ===

        # TODO: After identifying the missing predicate from exploration, update the domain/problem accordingly and rerun the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
