# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the environment's response.

        # For demonstration, we will try to use each available skill on each object/location and log the results.
        # This will help us infer which predicate is missing by observing which actions fail or succeed.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # If location names are not provided, try to infer from object positions
        if not location_names:
            # Assume locations are unique position tuples
            unique_locations = set()
            for pos in positions.values():
                unique_locations.add(tuple(pos))
            location_names = [f"location_{i}" for i in range(len(unique_locations))]

        # Try to execute each skill and catch exceptions to detect missing predicates
        exploration_results = {}
        for skill in available_skills:
            exploration_results[skill] = []
            for obj in object_names:
                for loc in location_names:
                    try:
                        # Dynamically call the skill if it exists
                        skill_fn = globals().get(skill, None)
                        if skill_fn is None:
                            continue
                        # Prepare arguments based on skill signature
                        if skill == 'execute_pick':
                            # Try picking the object at the location
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_place':
                            # Try placing the object at the location
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_push':
                            # Try pushing the object at the location
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_pull':
                            # Try pulling the object at the location
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_sweep':
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_rotate':
                            obs, reward, done = skill_fn(env, task, obj, loc)
                        elif skill == 'execute_go':
                            # Try moving from one location to another
                            for loc2 in location_names:
                                if loc != loc2:
                                    obs, reward, done = skill_fn(env, task, loc, loc2)
                        elif skill == 'execute_gripper':
                            obs, reward, done = skill_fn(env, task)
                        else:
                            continue
                        exploration_results[skill].append((obj, loc, "success"))
                    except Exception as e:
                        exploration_results[skill].append((obj, loc, f"fail: {str(e)}"))
                        continue

        # Analyze exploration results to infer missing predicate
        print("=== Exploration Results ===")
        for skill, results in exploration_results.items():
            for res in results:
                print(f"Skill: {skill}, Object: {res[0]}, Location: {res[1]}, Result: {res[2]}")

        print("=== Exploration phase complete. Please check above for failed actions and missing predicates. ===")

        # === (Optional) Main Task Plan Execution ===
        # After identifying the missing predicate, you would normally proceed to execute the oracle plan.
        # For this skeleton, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
