# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (not (need-ready)) ===
        # The feedback indicates that (not (need-ready)) is a missing predicate in the plan preconditions.
        # We will attempt to execute actions that require (not (need-ready)) as a precondition.
        # If the action fails, we will try to resolve the missing predicate by executing the appropriate skill.

        # For demonstration, we will attempt to pick an object from the floor.
        # This action requires (not (need-ready)) in its precondition.

        # Find an object on the floor and its location
        object_on_floor = None
        object_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_on_floor = obj_name
                object_location = obj_info.get('location', None)
                break

        if object_on_floor is None or object_location is None:
            print("[Exploration] No object on the floor found for exploration.")
        else:
            print(f"[Exploration] Attempting to pick object '{object_on_floor}' at location '{object_location}'.")

            # Try to execute the pick action
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=object_location
                )
                print("[Exploration] Pick action executed successfully.")
            except Exception as e:
                print(f"[Exploration] Pick action failed: {e}")
                print("[Exploration] Attempting to resolve missing (not (need-ready)) predicate.")

                # Try to execute the 'execute_go_ready' skill to clear the 'need-ready' predicate
                try:
                    # We need the robot's current location
                    robot_location = None
                    for pos_name, pos_info in positions.items():
                        if pos_info.get('is_robot', False):
                            robot_location = pos_info.get('location', None)
                            break
                    if robot_location is None:
                        # Fallback: try to get from task or obs
                        robot_location = getattr(task, 'robot_location', None)
                    if robot_location is None:
                        robot_location = 'ready-pose'  # Default fallback

                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_location=robot_location
                    )
                    print("[Exploration] Executed 'execute_go_ready' to clear 'need-ready'.")

                    # Retry the pick action
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=object_on_floor,
                        location=object_location
                    )
                    print("[Exploration] Pick action executed successfully after clearing 'need-ready'.")
                except Exception as e2:
                    print(f"[Exploration] Failed to resolve (not (need-ready)): {e2}")

        # === End of Exploration Phase ===

        # TODO: Continue with the rest of the oracle plan as needed.
        # For this exploration, we focus on identifying and resolving the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
