# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only way to clear (need-ready) is to execute the 'execute_go_ready' action.
        # Therefore, after any action that sets (need-ready), we must call the corresponding skill to clear it.

        # Since we do not have the oracle plan or the specific objects/locations, we will demonstrate the exploration
        # and handling of (need-ready) using the available skills and the domain logic.

        # === Example Exploration and Plan Execution ===

        # For demonstration, let's assume we have an object 'obj1' on the floor at location 'loc1',
        # and a drawer 'drawer1' at location 'loc2'. The robot starts at 'ready-pose'.
        # We'll pick up the object, place it in the drawer, and handle (need-ready) as required.

        # You should replace 'obj1', 'loc1', 'drawer1', 'loc2' with actual names from your environment.
        # For now, we will attempt to infer them from the positions dictionary.

        # Find an object on the floor and a drawer
        obj_name = None
        obj_pos = None
        drawer_name = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'handle' not in name and obj_name is None:
                obj_name = name
                obj_pos = pos

        # Fallback if not found
        if obj_name is None or drawer_name is None:
            print("[Exploration] Could not find suitable object or drawer in positions. Exiting.")
            return

        # Assume locations are named as 'loc1', 'loc2', etc.
        # For demonstration, we use the position keys as location names.
        loc_obj = obj_name + "_loc"
        loc_drawer = drawer_name + "_loc"
        ready_pose = "ready-pose"

        # For the sake of this example, let's define the location names as the object/drawer names.
        # In a real environment, you would map positions to location names.

        # 1. Move to the object's location (if not already there)
        print(f"[Task] Moving to object '{obj_name}' at location '{loc_obj}'")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=loc_obj,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to move to object location: {e}")
            return

        # 2. Pick the object from the floor (sets need-ready)
        print(f"[Task] Picking up object '{obj_name}' from floor at '{loc_obj}'")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=obj_name,
                location=loc_obj,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 3. After pick, (need-ready) is set. Must clear it by going to ready-pose.
        print("[Task] Executing go_ready to clear (need-ready)")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=loc_obj,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to execute go_ready: {e}")
            return

        # 4. Move to the drawer location
        print(f"[Task] Moving to drawer '{drawer_name}' at location '{loc_drawer}'")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=loc_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")
            return

        # 5. Place the object into the drawer (sets need-ready)
        print(f"[Task] Placing object '{obj_name}' into drawer '{drawer_name}' at '{loc_drawer}'")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=obj_name,
                drawer_name=drawer_name,
                location=loc_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to place object in drawer: {e}")
            return

        # 6. After place, (need-ready) is set. Must clear it by going to ready-pose.
        print("[Task] Executing go_ready to clear (need-ready) after place")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=loc_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to execute go_ready after place: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()