# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will perform an exploration using only the available skills to check and ensure the robot is in a "free" state.
        # The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # The predicate (robot-free) is set by actions such as execute_pick, execute_place, execute_pull, and is a precondition for execute_go, execute_sweep, execute_gripper, and execute_push.
        # We will attempt to use these skills in a minimal exploration sequence to check if the robot can reach the "free" state.

        # For demonstration, we will:
        # 1. Try to use execute_gripper (which requires robot-free as precondition)
        # 2. If not possible, try to use execute_push or execute_sweep
        # 3. If still not possible, attempt to use execute_place or execute_pick to reach robot-free

        # Note: The actual arguments for these skills depend on the environment and objects.
        # We'll attempt to find suitable objects/locations from positions.

        # Helper: Find a location and an object on the floor (if any)
        location_name = None
        object_name = None
        for name, pos in positions.items():
            if 'location' in name or 'room' in name or 'table' in name:
                location_name = name
            if 'object' in name or 'ball' in name or 'cube' in name:
                object_name = name
            if location_name and object_name:
                break

        # If not found, just pick any available keys
        if not location_name and positions:
            location_name = list(positions.keys())[0]
        if not object_name and positions:
            object_name = list(positions.keys())[0]

        # 1. Try execute_gripper (no parameters)
        try:
            print("[Exploration] Trying execute_gripper to check robot-free state...")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper executed successfully.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)
            # 2. Try execute_sweep (requires object, location)
            try:
                print("[Exploration] Trying execute_sweep...")
                obs, reward, done = execute_sweep(env, task, object_name, location_name)
                print("[Exploration] execute_sweep executed successfully.")
            except Exception as e2:
                print("[Exploration] execute_sweep failed:", e2)
                # 3. Try execute_push (requires drawer, location)
                # Try to find a drawer in positions
                drawer_name = None
                for name in positions.keys():
                    if 'drawer' in name:
                        drawer_name = name
                        break
                if drawer_name:
                    try:
                        print("[Exploration] Trying execute_push...")
                        obs, reward, done = execute_push(env, task, drawer_name, location_name)
                        print("[Exploration] execute_push executed successfully.")
                    except Exception as e3:
                        print("[Exploration] execute_push failed:", e3)
                else:
                    print("[Exploration] No drawer found for execute_push.")
                # 4. Try execute_place or execute_pick to reach robot-free
                try:
                    print("[Exploration] Trying execute_pick...")
                    obs, reward, done = execute_pick(env, task, object_name, location_name)
                    print("[Exploration] execute_pick executed successfully.")
                except Exception as e4:
                    print("[Exploration] execute_pick failed:", e4)
                # Try execute_place if possible
                try:
                    print("[Exploration] Trying execute_place...")
                    # For execute_place, need object, drawer, location
                    if drawer_name:
                        obs, reward, done = execute_place(env, task, object_name, drawer_name, location_name)
                        print("[Exploration] execute_place executed successfully.")
                    else:
                        print("[Exploration] No drawer found for execute_place.")
                except Exception as e5:
                    print("[Exploration] execute_place failed:", e5)

        # After exploration, print a message about the missing predicate
        print("[Exploration] If all above actions failed, the predicate (robot-free) may be missing or not being set correctly in the environment.")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan Execution ===
        # At this point, you would continue with the main plan steps to achieve the task goal.
        # For this code, the focus is on the exploration phase for the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()