# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects, drawers, and locations from positions
        # (Assume positions dict contains keys for objects, drawers, and locations)
        objects = [k for k in positions if 'object' in k]
        drawers = [k for k in positions if 'drawer' in k]
        handles = [k for k in positions if 'handle' in k]
        locations = [k for k in positions if 'location' in k or 'room' in k or 'pos' in k]

        # For robust exploration, try all available skills on all objects/drawers/locations
        # and catch any exceptions to infer missing predicates

        # Track which predicates might be missing
        missing_predicates = set()

        # Try to pick up each object from the floor
        for obj in objects:
            try:
                print(f"[Exploration] Trying to pick up {obj} at its position.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    positions[obj]
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                else:
                    print(f"[Exploration] Could not pick up {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick on {obj}: {e}")
                missing_predicates.add('on-floor or hand-empty or robot-free or robot-at')

        # Try to open each drawer using its handle
        for drawer in drawers:
            # Find handle for this drawer
            handle = None
            for h in handles:
                if f"handle-of_{h}_{drawer}" in positions or f"{h}_of_{drawer}" in positions:
                    handle = h
                    break
            if handle is None and handles:
                handle = handles[0]  # fallback: just pick the first handle
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to pull open {drawer} with handle {handle} at {loc}.")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer,
                        handle,
                        positions[loc]
                    )
                    if done:
                        print(f"[Exploration] Successfully pulled open {drawer}.")
                    else:
                        print(f"[Exploration] Could not pull open {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pull on {drawer}: {e}")
                    missing_predicates.add('drawer-locked or drawer-unlocked or drawer-closed or handle-of or holding')

        # Try to place each object into each drawer at each location
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying to place {obj} into {drawer} at {loc}.")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj,
                            drawer,
                            positions[loc]
                        )
                        if done:
                            print(f"[Exploration] Successfully placed {obj} into {drawer}.")
                        else:
                            print(f"[Exploration] Could not place {obj} into {drawer}.")
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_place on {obj} into {drawer}: {e}")
                        missing_predicates.add('drawer-open or drawer-full or holding or robot-at')

        # Try to push (close) each drawer at each location
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to push (close) {drawer} at {loc}.")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer,
                        positions[loc]
                    )
                    if done:
                        print(f"[Exploration] Successfully pushed (closed) {drawer}.")
                    else:
                        print(f"[Exploration] Could not push (close) {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_push on {drawer}: {e}")
                    missing_predicates.add('drawer-open or robot-free or robot-at')

        # Try to move (go) between all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Trying to go from {from_loc} to {to_loc}.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        positions[from_loc],
                        positions[to_loc]
                    )
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                    else:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go from {from_loc} to {to_loc}: {e}")
                    missing_predicates.add('robot-free or robot-at')

        # Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}.")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj,
                        positions[loc]
                    )
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}.")
                    else:
                        print(f"[Exploration] Could not sweep {obj} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_sweep on {obj} at {loc}: {e}")
                    missing_predicates.add('on-floor or robot-free or robot-at')

        # Try to use gripper (if applicable)
        try:
            print(f"[Exploration] Trying to use gripper.")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
            else:
                print(f"[Exploration] Could not use gripper.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")
            missing_predicates.add('robot-free')

        # === Exploration Summary ===
        if missing_predicates:
            print("\n[Exploration] Potential missing predicates detected:")
            for pred in missing_predicates:
                print(f"  - {pred}")
        else:
            print("\n[Exploration] No missing predicates detected during skill execution.")

        # === End of Exploration Phase ===
        # At this point, you can use the exploration results to update your domain/problem files or inform the planner.

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would execute it here step by step using the available skills.
        # For this exploration-focused code, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()