# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects, drawers, and locations from positions (assuming keys are descriptive)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # For demonstration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to open each drawer (pull)
        # 4. Try to place an object in a drawer (place)
        # 5. Try to push (close) each drawer

        # We will log any failures or exceptions to help identify missing predicates.

        # Helper: get current robot location (if available)
        def get_robot_location():
            # Try to infer robot location from observation or positions
            for loc in location_names:
                if positions.get('robot_at', None) == loc:
                    return loc
            # Fallback: pick the first location
            if location_names:
                return location_names[0]
            return None

        # 1. Move to each location
        robot_location = get_robot_location()
        for loc in location_names:
            if robot_location != loc:
                try:
                    print(f"[Exploration] Moving robot from {robot_location} to {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration][Error] Failed to move to {loc}: {e}")

        # 2. Try to pick each object
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to pick object {obj} at {robot_location}")
                obs, reward, done = execute_pick(env, task, obj, robot_location)
            except Exception as e:
                print(f"[Exploration][Error] Failed to pick {obj}: {e}")

        # 3. Try to open (pull) each drawer using its handle (if handle known)
        for drawer in drawer_names:
            # Try to find a handle for this drawer
            handle_name = None
            for obj in object_names:
                if f"handle_of_{drawer}" in obj or f"{drawer}_handle" in obj or "handle" in obj:
                    handle_name = obj
                    break
            if not handle_name:
                # Try generic handle
                for obj in object_names:
                    if "handle" in obj:
                        handle_name = obj
                        break
            if not handle_name:
                print(f"[Exploration][Warning] No handle found for drawer {drawer}, skipping pull.")
                continue
            try:
                print(f"[Exploration] Attempting to pick handle {handle_name} for drawer {drawer}")
                obs, reward, done = execute_pick(env, task, handle_name, robot_location)
            except Exception as e:
                print(f"[Exploration][Error] Failed to pick handle {handle_name}: {e}")
            try:
                print(f"[Exploration] Attempting to pull (open) drawer {drawer} with handle {handle_name}")
                obs, reward, done = execute_pull(env, task, drawer, handle_name, robot_location)
            except Exception as e:
                print(f"[Exploration][Error] Failed to pull (open) drawer {drawer}: {e}")

        # 4. Try to place each object in each drawer
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Attempting to place object {obj} in drawer {drawer} at {robot_location}")
                    obs, reward, done = execute_place(env, task, obj, drawer, robot_location)
                except Exception as e:
                    print(f"[Exploration][Error] Failed to place {obj} in {drawer}: {e}")

        # 5. Try to push (close) each drawer
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to push (close) drawer {drawer} at {robot_location}")
                obs, reward, done = execute_push(env, task, drawer, robot_location)
            except Exception as e:
                print(f"[Exploration][Error] Failed to push (close) drawer {drawer}: {e}")

        # 6. Try to sweep each object (if available)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to sweep object {obj} at {robot_location}")
                obs, reward, done = execute_sweep(env, task, obj, robot_location)
            except Exception as e:
                print(f"[Exploration][Error] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration][Error] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()