# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (robot-free) ===
        # Feedback indicates (robot-free) is missing or not being set as expected.
        # We will explore the environment and attempt to trigger all skills that may affect robot-free.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to execute each skill in a safe, exploratory manner to observe their effects.
        # This will help us determine which skill(s) set or unset the (robot-free) predicate.

        # For this, we need to select sample objects, drawers, handles, and locations from the environment.
        # We'll use the positions dictionary to get available objects and locations.

        # Extract sample objects and locations
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'pos' in name]

        # Fallbacks in case names are not as expected
        if not object_names:
            object_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())

        # Select first available object, drawer, handle, and two locations for exploration
        obj = object_names[0] if object_names else None
        drawer = drawer_names[0] if drawer_names else None
        handle = handle_names[0] if handle_names else None
        loc1 = location_names[0] if location_names else None
        loc2 = location_names[1] if len(location_names) > 1 else loc1

        print("[Exploration] Using object:", obj)
        print("[Exploration] Using drawer:", drawer)
        print("[Exploration] Using handle:", handle)
        print("[Exploration] Using locations:", loc1, loc2)

        # 1. Try execute_go (move robot between two locations)
        try:
            print("[Exploration] Trying execute_go...")
            obs, reward, done = execute_go(env, task, loc1, loc2)
            print("[Exploration] execute_go completed.")
        except Exception as e:
            print("[Exploration] execute_go failed:", e)

        # 2. Try execute_pick (pick up an object)
        try:
            print("[Exploration] Trying execute_pick...")
            obs, reward, done = execute_pick(env, task, obj, loc2)
            print("[Exploration] execute_pick completed.")
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)

        # 3. Try execute_place (place object in drawer)
        try:
            print("[Exploration] Trying execute_place...")
            obs, reward, done = execute_place(env, task, obj, drawer, loc2)
            print("[Exploration] execute_place completed.")
        except Exception as e:
            print("[Exploration] execute_place failed:", e)

        # 4. Try execute_pull (open drawer using handle)
        try:
            print("[Exploration] Trying execute_pull...")
            obs, reward, done = execute_pull(env, task, drawer, handle, loc2)
            print("[Exploration] execute_pull completed.")
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)

        # 5. Try execute_push (close drawer)
        try:
            print("[Exploration] Trying execute_push...")
            obs, reward, done = execute_push(env, task, drawer, loc2)
            print("[Exploration] execute_push completed.")
        except Exception as e:
            print("[Exploration] execute_push failed:", e)

        # 6. Try execute_sweep (sweep object)
        try:
            print("[Exploration] Trying execute_sweep...")
            obs, reward, done = execute_sweep(env, task, obj, loc2)
            print("[Exploration] execute_sweep completed.")
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)

        # 7. Try execute_rotate (if available)
        if 'execute_rotate' in globals():
            try:
                print("[Exploration] Trying execute_rotate...")
                obs, reward, done = execute_rotate(env, task, obj, loc2)
                print("[Exploration] execute_rotate completed.")
            except Exception as e:
                print("[Exploration] execute_rotate failed:", e)
        else:
            print("[Exploration] execute_rotate not available.")

        # 8. Try execute_gripper (if available)
        if 'execute_gripper' in globals():
            try:
                print("[Exploration] Trying execute_gripper...")
                obs, reward, done = execute_gripper(env, task)
                print("[Exploration] execute_gripper completed.")
            except Exception as e:
                print("[Exploration] execute_gripper failed:", e)
        else:
            print("[Exploration] execute_gripper not available.")

        # After each action, you may want to check the state of the environment to see if (robot-free) is set/unset.
        # This can be done by inspecting the observation or task state if such an interface is available.
        # For now, we print the observation after each action for manual inspection.

        print("[Exploration] Exploration phase complete. Check logs for predicate (robot-free) status.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()