# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (drawer-closed drawer1) is missing or needs to be checked.
        # We need to explore the state of drawer1 to determine if it is closed.
        # Use only available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this example, let's assume the following object names and locations:
        # - drawer1: the drawer to be explored
        # - handle1: the handle of drawer1
        # - robot starts at 'start_pos'
        # - drawer1 is at 'drawer_pos'
        # - handle1 is at 'handle_pos'
        # These names should match those in your environment and object_positions.

        # You may need to adjust these names based on your environment setup.
        try:
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            robot_start = 'start_pos'
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            robot_pos = positions.get('robot', None)
        except Exception as e:
            print("[Error] Could not retrieve object positions:", e)
            drawer_pos = None
            handle_pos = None
            robot_pos = None

        # Step 1: Move robot to the drawer location if not already there
        try:
            if robot_pos is not None and drawer_pos is not None and robot_pos != drawer_pos:
                print(f"[Exploration] Moving robot from {robot_pos} to {drawer_pos}")
                obs, reward, done = execute_go(env, task, robot_pos, drawer_pos)
                robot_pos = drawer_pos
        except Exception as e:
            print("[Error] Failed to move robot during exploration:", e)

        # Step 2: Try to pick the handle (if required by your domain)
        # This is to prepare for pulling the drawer to check its state
        try:
            if handle_pos is not None and robot_pos == handle_pos:
                print(f"[Exploration] Picking up handle {handle_name} at {handle_pos}")
                obs, reward, done = execute_pick(env, task, handle_name, handle_pos)
        except Exception as e:
            print("[Error] Failed to pick handle during exploration:", e)

        # Step 3: Try to pull the drawer to check if it is closed or locked
        # This will help us determine if (drawer-closed drawer1) is true/missing
        try:
            print(f"[Exploration] Attempting to pull {drawer_name} using {handle_name} at {drawer_pos}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_pos)
            # If the pull succeeds, the drawer was closed and is now open
            print("[Exploration] Drawer pull succeeded. Drawer was closed and is now open.")
        except Exception as e:
            print("[Exploration] Drawer pull failed. Drawer may not be closed or is locked:", e)
            # This suggests the predicate (drawer-closed drawer1) may be missing or false

        # === End of Exploration Phase ===

        # === Main Task Plan ===
        # At this point, you can proceed with the oracle plan using the available skills.
        # For example, if the goal is to place an object in the drawer, you would:
        # 1. Pick up the object from the floor
        # 2. Move to the drawer
        # 3. Open the drawer if needed
        # 4. Place the object inside

        # Example plan (replace object names/locations as needed):
        try:
            # 1. Pick up object from floor
            object_to_pick = 'object1'
            object_pos = positions.get(object_to_pick, None)
            if object_pos is not None and robot_pos != object_pos:
                print(f"[Task] Moving robot to {object_pos} to pick up {object_to_pick}")
                obs, reward, done = execute_go(env, task, robot_pos, object_pos)
                robot_pos = object_pos
            if object_pos is not None:
                print(f"[Task] Picking up {object_to_pick} at {object_pos}")
                obs, reward, done = execute_pick(env, task, object_to_pick, object_pos)

            # 2. Move to drawer
            if robot_pos != drawer_pos and drawer_pos is not None:
                print(f"[Task] Moving robot to drawer at {drawer_pos}")
                obs, reward, done = execute_go(env, task, robot_pos, drawer_pos)
                robot_pos = drawer_pos

            # 3. Open drawer if not already open (already attempted in exploration)
            # 4. Place object in drawer
            print(f"[Task] Placing {object_to_pick} in {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_place(env, task, object_to_pick, drawer_name, drawer_pos)

            # 5. Optionally, close the drawer
            print(f"[Task] Closing {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_pos)

        except Exception as e:
            print("[Error] Exception during main task plan execution:", e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
