# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects, drawers, and locations from positions
        # (Assume positions dict contains keys like 'objects', 'drawers', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        handles = positions.get('handles', [])

        # Get robot's initial location
        robot_location = positions.get('robot', None)
        if robot_location is None and len(locations) > 0:
            robot_location = locations[0]

        # Try to explore each drawer and object
        for drawer in drawers:
            # Try to go to the drawer's location
            drawer_location = positions.get(f'{drawer}_location', None)
            if drawer_location is None and len(locations) > 0:
                drawer_location = locations[0]
            try:
                # Move to drawer location if not already there
                if robot_location != drawer_location:
                    print(f"[Exploration] Moving to {drawer_location} to explore {drawer}")
                    obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                    robot_location = drawer_location
            except Exception as e:
                print(f"[Exploration] Failed to move to {drawer_location}: {e}")

            # Try to pull the drawer using its handle (if any)
            handle = None
            for h in handles:
                if positions.get(f'handle_of_{h}', None) == drawer:
                    handle = h
                    break
            if handle:
                try:
                    # Pick the handle
                    print(f"[Exploration] Attempting to pick handle {handle} of drawer {drawer}")
                    obs, reward, done = execute_pick(env, task, handle, drawer_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pick handle {handle}: {e}")
                try:
                    # Pull the drawer
                    print(f"[Exploration] Attempting to pull drawer {drawer} using handle {handle}")
                    obs, reward, done = execute_pull(env, task, drawer, handle, drawer_location)
                except Exception as e:
                    print(f"[Exploration] Failed to pull drawer {drawer}: {e}")
            else:
                print(f"[Exploration] No handle found for drawer {drawer}")

        # Try to pick and place each object
        for obj in objects:
            obj_location = positions.get(f'{obj}_location', None)
            if obj_location is None and len(locations) > 0:
                obj_location = locations[0]
            try:
                # Move to object location if not already there
                if robot_location != obj_location:
                    print(f"[Exploration] Moving to {obj_location} to explore object {obj}")
                    obs, reward, done = execute_go(env, task, robot_location, obj_location)
                    robot_location = obj_location
            except Exception as e:
                print(f"[Exploration] Failed to move to {obj_location}: {e}")

            try:
                # Try to pick the object
                print(f"[Exploration] Attempting to pick object {obj}")
                obs, reward, done = execute_pick(env, task, obj, obj_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick object {obj}: {e}")

            # Try to place the object in each drawer
            for drawer in drawers:
                drawer_location = positions.get(f'{drawer}_location', None)
                if drawer_location is None and len(locations) > 0:
                    drawer_location = locations[0]
                try:
                    # Move to drawer location if not already there
                    if robot_location != drawer_location:
                        print(f"[Exploration] Moving to {drawer_location} to place object {obj} in drawer {drawer}")
                        obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                        robot_location = drawer_location
                except Exception as e:
                    print(f"[Exploration] Failed to move to {drawer_location}: {e}")

                try:
                    print(f"[Exploration] Attempting to place object {obj} in drawer {drawer}")
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_location)
                except Exception as e:
                    print(f"[Exploration] Failed to place object {obj} in drawer {drawer}: {e}")

        # Try to push (close) all drawers
        for drawer in drawers:
            drawer_location = positions.get(f'{drawer}_location', None)
            if drawer_location is None and len(locations) > 0:
                drawer_location = locations[0]
            try:
                # Move to drawer location if not already there
                if robot_location != drawer_location:
                    print(f"[Exploration] Moving to {drawer_location} to push drawer {drawer}")
                    obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                    robot_location = drawer_location
            except Exception as e:
                print(f"[Exploration] Failed to move to {drawer_location}: {e}")

            try:
                print(f"[Exploration] Attempting to push (close) drawer {drawer}")
                obs, reward, done = execute_push(env, task, drawer, drawer_location)
            except Exception as e:
                print(f"[Exploration] Failed to push drawer {drawer}: {e}")

        # Try to sweep all objects (if available)
        for obj in objects:
            obj_location = positions.get(f'{obj}_location', None)
            if obj_location is None and len(locations) > 0:
                obj_location = locations[0]
            try:
                # Move to object location if not already there
                if robot_location != obj_location:
                    print(f"[Exploration] Moving to {obj_location} to sweep object {obj}")
                    obs, reward, done = execute_go(env, task, robot_location, obj_location)
                    robot_location = obj_location
            except Exception as e:
                print(f"[Exploration] Failed to move to {obj_location}: {e}")

            try:
                print(f"[Exploration] Attempting to sweep object {obj}")
                obs, reward, done = execute_sweep(env, task, obj, obj_location)
            except Exception as e:
                print(f"[Exploration] Failed to sweep object {obj}: {e}")

        # Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()