# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ---
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We will use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions for this code:
        # - There is a drawer named 'drawer1' and a handle object (e.g., 'handle1') attached to it.
        # - The robot starts at some initial location, e.g., 'loc_robot_start'.
        # - The handle object is on the floor near the drawer, or its position is known.
        # - The positions dictionary contains keys for 'drawer1', 'handle1', and relevant locations.

        # --- Step 1: Move to the handle's location ---
        try:
            handle_name = None
            drawer_name = None
            handle_pos = None
            drawer_pos = None
            robot_pos = None

            # Try to infer handle and drawer names from positions
            for obj_name in positions:
                if 'handle' in obj_name:
                    handle_name = obj_name
                    handle_pos = positions[obj_name]
                if 'drawer' in obj_name:
                    drawer_name = obj_name
                    drawer_pos = positions[obj_name]
                if 'robot' in obj_name:
                    robot_pos = positions[obj_name]

            if handle_name is None or drawer_name is None or handle_pos is None or drawer_pos is None:
                print("[Exploration] Could not find handle or drawer in object positions. Exploration aborted.")
                return

            # If robot position is not given, assume a default location
            if robot_pos is None:
                robot_pos = positions.get('robot', None)
                if robot_pos is None:
                    # Fallback: use a default location string if available
                    robot_pos = 'loc_robot_start'

            # Move to handle location (if not already there)
            # Assume location names are available as keys or as part of the position info
            # For this code, we use the handle's location as the target
            print(f"[Exploration] Moving robot to handle location: {handle_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_pos,
                to_location=handle_pos
            )
            if done:
                print("[Exploration] Task ended unexpectedly during move to handle.")
                return

            # --- Step 2: Pick up the handle object ---
            print(f"[Exploration] Picking up handle object: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_pos
            )
            if done:
                print("[Exploration] Task ended unexpectedly during pick.")
                return

            # --- Step 3: Attempt to pull the drawer using the handle ---
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_pos
            )
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return

            # --- Step 4: Check if the drawer is now open (i.e., unlocked) ---
            # In a real system, we would check the environment state or observation for the predicate
            # For this code, we print a message indicating the exploration is complete
            print(f"[Exploration] Pulled the drawer. Check if predicate (drawer-unlocked {drawer_name}) is now true in the environment.")

        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")

        # --- End of Exploration Phase ---

        # TODO: Continue with the main oracle plan after exploration, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()