# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and State Information ===
        positions = get_object_positions()
        # Example: positions = {'dice1': (x1, y1, z1), 'drawer1': (x2, y2, z2), ...}

        # === Define object and location names based on the problem ===
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle1', 'handle2', 'handle3']
        drawer_names = ['drawer1', 'drawer2', 'drawer3']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # === Exploration Phase: Identify Drawer States ===
        # The feedback requires us to explicitly determine which drawer is open/closed/unlocked/locked.
        # We'll use the available skills and the environment's state to do this.

        # We'll keep track of drawer states
        drawer_states = {}
        open_drawer = None
        closed_drawers = []
        unlocked_drawers = []
        locked_drawers = []

        # Try to determine the state of each drawer
        for drawer in drawer_names:
            # Try to check if the drawer is open or closed
            # We assume get_object_positions() or task.get_observation() provides this info
            # If not, we must explore using available skills

            # Try to get state from observation
            obs = task.get_observation()
            is_open = False
            is_closed = False
            is_unlocked = False
            is_locked = False

            # Check for open/closed/unlocked/locked predicates in observation
            # (Assume obs['predicates'] is a dict of predicate: [objects])
            predicates = obs.get('predicates', {})

            if 'drawer-open' in predicates and drawer in predicates['drawer-open']:
                is_open = True
            if 'drawer-closed' in predicates and drawer in predicates['drawer-closed']:
                is_closed = True
            if 'drawer-unlocked' in predicates and drawer in predicates['drawer-unlocked']:
                is_unlocked = True
            if 'drawer-locked' in predicates and drawer in predicates['drawer-locked']:
                is_locked = True

            # If not available, try to infer by attempting to interact (exploration)
            if not (is_open or is_closed):
                # Try to use execute_pull or execute_push to test state
                # First, find the handle for this drawer
                handle = None
                for h in handle_names:
                    # Assume handle-of relation is in obs['predicates']
                    if 'handle-of' in predicates and [h, drawer] in predicates['handle-of']:
                        handle = h
                        break
                if handle is None:
                    # Fallback: assume handle name matches drawer index
                    idx = drawer_names.index(drawer)
                    handle = handle_names[idx]

                # Try to pick the handle if possible
                try:
                    # Move to handle location if needed
                    handle_pos = positions.get(handle, None)
                    if handle_pos is not None:
                        # Move robot to handle location
                        robot_loc = None
                        for loc in location_names:
                            if 'robot-at' in predicates and loc in predicates['robot-at']:
                                robot_loc = loc
                                break
                        if robot_loc is not None and robot_loc != handle_pos:
                            # Use execute_go to move
                            try:
                                obs, reward, done = execute_go(env, task, robot_loc, handle_pos)
                            except Exception as e:
                                pass
                        # Try to pick the handle
                        try:
                            obs, reward, done = execute_pick(env, task, handle, handle_pos)
                        except Exception as e:
                            pass
                        # Try to pull the handle (open drawer)
                        try:
                            obs, reward, done = execute_pull(env, task, drawer, handle, handle_pos)
                            # If successful, drawer is now open
                            is_open = True
                        except Exception as e:
                            # If fails, drawer may be locked or already open
                            pass
                except Exception as e:
                    pass

            # Record state
            drawer_states[drawer] = {
                'open': is_open,
                'closed': is_closed,
                'unlocked': is_unlocked,
                'locked': is_locked
            }
            if is_open:
                open_drawer = drawer
            if is_closed:
                closed_drawers.append(drawer)
            if is_unlocked:
                unlocked_drawers.append(drawer)
            if is_locked:
                locked_drawers.append(drawer)

        # If we still don't know which drawer is open, pick the first open one by default
        if open_drawer is None:
            for drawer, state in drawer_states.items():
                if state['open']:
                    open_drawer = drawer
                    break
            if open_drawer is None:
                # Fallback: pick drawer1 as open
                open_drawer = 'drawer1'

        # === Main Task Plan: Insert Dice into the Open Drawer ===
        # For each dice, pick it up and place it in the open drawer

        for dice in dice_names:
            try:
                # Get dice position
                dice_pos = positions.get(dice, None)
                if dice_pos is None:
                    # Fallback: use 'floor'
                    dice_pos = 'floor'
                # Move to dice location if needed
                robot_loc = None
                obs = task.get_observation()
                predicates = obs.get('predicates', {})
                for loc in location_names:
                    if 'robot-at' in predicates and loc in predicates['robot-at']:
                        robot_loc = loc
                        break
                if robot_loc is not None and robot_loc != dice_pos:
                    try:
                        obs, reward, done = execute_go(env, task, robot_loc, dice_pos)
                    except Exception as e:
                        pass
                # Pick up the dice
                try:
                    obs, reward, done = execute_pick(env, task, dice, dice_pos)
                except Exception as e:
                    continue  # Skip to next dice if pick fails

                # Move to open drawer location
                drawer_pos = positions.get(open_drawer, None)
                if drawer_pos is None:
                    # Fallback: use 'drawer-area'
                    drawer_pos = 'drawer-area'
                obs = task.get_observation()
                predicates = obs.get('predicates', {})
                robot_loc = None
                for loc in location_names:
                    if 'robot-at' in predicates and loc in predicates['robot-at']:
                        robot_loc = loc
                        break
                if robot_loc is not None and robot_loc != drawer_pos:
                    try:
                        obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                    except Exception as e:
                        pass

                # Place the dice in the open drawer
                try:
                    obs, reward, done = execute_place(env, task, dice, open_drawer, drawer_pos)
                except Exception as e:
                    continue  # Skip to next dice if place fails

            except Exception as e:
                print(f"[Error] Failed to process {dice}: {e}")
                continue

        # === Optionally, close the drawer after placing all dice ===
        try:
            drawer_pos = positions.get(open_drawer, None)
            if drawer_pos is None:
                drawer_pos = 'drawer-area'
            obs = task.get_observation()
            predicates = obs.get('predicates', {})
            robot_loc = None
            for loc in location_names:
                if 'robot-at' in predicates and loc in predicates['robot-at']:
                    robot_loc = loc
                    break
            if robot_loc is not None and robot_loc != drawer_pos:
                try:
                    obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                except Exception as e:
                    pass
            # Only push if drawer is open and robot is free
            if open_drawer is not None:
                try:
                    obs, reward, done = execute_push(env, task, open_drawer, drawer_pos)
                except Exception as e:
                    pass
        except Exception as e:
            pass

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()