# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and PDDL) ===
        object_names = [
            "dice1", "dice2",
            "handle1", "handle2", "handle3",
            "drawer1", "drawer2", "drawer3"
        ]
        location_names = [
            "floor", "drawer-area", "ready-pose"
        ]

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to discover a missing predicate,
        # such as 'lock-known' for a drawer (e.g., drawer2 is locked, but this may not be known initially).
        # We'll perform an exploration step to ensure the robot knows the lock state before attempting to open.

        # For this, we simulate the robot moving to the handle of drawer2 and attempting to pull it,
        # which will reveal if the drawer is locked (lock-known predicate).

        # --- Step 1: Move to handle2 (handle for drawer2) ---
        try:
            # Move from ready-pose to floor (where handle2 is located)
            obs, reward, done = execute_go(
                env,
                task,
                from_location="ready-pose",
                to_location="floor"
            )
        except Exception as e:
            print("[Error] Failed to move to floor for handle2:", e)
            return

        # --- Step 2: Pick up handle2 ---
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name="handle2",
                location_name="floor"
            )
        except Exception as e:
            print("[Error] Failed to pick up handle2:", e)
            return

        # --- Step 3: Attempt to pull handle2 (exploration: discover lock state) ---
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name="drawer2",
                handle_name="handle2",
                location_name="floor"
            )
            # If the drawer is locked, this should reveal the lock state (lock-known)
            print("[Exploration] Pulled handle2 to check lock state of drawer2.")
        except Exception as e:
            print("[Exploration] Could not pull handle2 (drawer2 may be locked):", e)
            # The failure here is expected if the drawer is locked and not yet unlocked

        # --- Step 4: Place handle2 back on the floor (if still holding) ---
        try:
            # If the robot is still holding handle2, place it back
            obs, reward, done = execute_place(
                env,
                task,
                object_name="handle2",
                drawer_name=None,  # Not placing in a drawer, just releasing
                location_name="floor"
            )
        except Exception as e:
            # It's okay if not holding, just continue
            pass

        # === Main Task: Put dice1 and dice2 into drawer1 (which is open and empty) ===

        # --- Step 5: Move to dice1 ---
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location="floor",
                to_location="floor"
            )
        except Exception as e:
            print("[Error] Failed to move to dice1:", e)
            return

        # --- Step 6: Pick up dice1 ---
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name="dice1",
                location_name="floor"
            )
        except Exception as e:
            print("[Error] Failed to pick up dice1:", e)
            return

        # --- Step 7: Move to drawer1 (drawer-area) ---
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location="floor",
                to_location="drawer-area"
            )
        except Exception as e:
            print("[Error] Failed to move to drawer-area for dice1:", e)
            return

        # --- Step 8: Place dice1 into drawer1 ---
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name="dice1",
                drawer_name="drawer1",
                location_name="drawer-area"
            )
        except Exception as e:
            print("[Error] Failed to place dice1 into drawer1:", e)
            return

        # --- Step 9: Move to dice2 ---
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location="drawer-area",
                to_location="floor"
            )
        except Exception as e:
            print("[Error] Failed to move to dice2:", e)
            return

        # --- Step 10: Pick up dice2 ---
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name="dice2",
                location_name="floor"
            )
        except Exception as e:
            print("[Error] Failed to pick up dice2:", e)
            return

        # --- Step 11: Move to drawer1 (drawer-area) ---
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location="floor",
                to_location="drawer-area"
            )
        except Exception as e:
            print("[Error] Failed to move to drawer-area for dice2:", e)
            return

        # --- Step 12: Place dice2 into drawer1 ---
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name="dice2",
                drawer_name="drawer1",
                location_name="drawer-area"
            )
        except Exception as e:
            print("[Error] Failed to place dice2 into drawer1:", e)
            return

        # --- Step 13: (Optional) Push drawer1 closed ---
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name="drawer1",
                location_name="drawer-area"
            )
        except Exception as e:
            # It's okay if the drawer is already closed or cannot be closed
            pass

        print("[Task] Successfully placed dice1 and dice2 into drawer1.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()