# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'dice1': (x1,y1,z1), 'dice2': (x2,y2,z2), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to find missing predicates (e.g., robot-free, hand-empty, etc.)
        # We'll use available skills to probe the environment and print out what we find.

        # 1. List all objects and locations from positions
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if name.startswith("dice") or name.startswith("object") or name.startswith("handle") or name.startswith("drawer"):
                object_names.append(name)
            if isinstance(pos, dict) and "location" in pos:
                location_names.add(pos["location"])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # If positions are 3D, we can't infer location names directly
                pass

        # Fallback: If no location names, use some defaults
        if not location_names:
            location_names = {"table", "drawer_area", "floor"}

        # 2. Try to use exploration skills to probe for missing predicates
        # For this, we attempt to use execute_go, execute_pick, etc., and catch any errors or unexpected states

        # We'll try to pick up each object and see if any predicate is missing (e.g., robot-free, hand-empty)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick up {obj}...")
                # Try to pick up the object at its position
                # We need to know the location; if not available, use a default
                obj_pos = positions.get(obj, None)
                if obj_pos is None:
                    print(f"  [Warning] No position for {obj}, skipping.")
                    continue
                # For this example, we assume all objects are on the 'floor'
                location = "floor"
                # Try to execute the pick skill
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=obj,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"  [Success] Picked up {obj}.")
                # Try to place it in a drawer if possible
                for drawer in object_names:
                    if "drawer" in drawer:
                        print(f"  [Exploration] Attempting to place {obj} in {drawer}...")
                        drawer_pos = positions.get(drawer, None)
                        if drawer_pos is None:
                            continue
                        obs, reward, done = execute_place(
                            env,
                            task,
                            target_obj=obj,
                            target_drawer=drawer,
                            target_pos=drawer_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"    [Success] Placed {obj} in {drawer}.")
                        break
                # After placing, try to push (close) the drawer
                for drawer in object_names:
                    if "drawer" in drawer:
                        print(f"  [Exploration] Attempting to push (close) {drawer}...")
                        drawer_pos = positions.get(drawer, None)
                        if drawer_pos is None:
                            continue
                        obs, reward, done = execute_push(
                            env,
                            task,
                            target_drawer=drawer,
                            target_pos=drawer_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"    [Success] Pushed (closed) {drawer}.")
                        break
            except Exception as e:
                print(f"  [Error] Could not pick/place/push {obj}: {e}")

        # 3. Try to move the robot between locations using execute_go
        location_list = list(location_names)
        if len(location_list) >= 2:
            from_loc = location_list[0]
            to_loc = location_list[1]
            try:
                print(f"[Exploration] Attempting to move robot from {from_loc} to {to_loc}...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"  [Success] Robot moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"  [Error] Could not move robot: {e}")

        # 4. Try to pull a handle (if any) to open a drawer and check for lock-known predicate
        for handle in object_names:
            if "handle" in handle:
                try:
                    print(f"[Exploration] Attempting to pull {handle}...")
                    handle_pos = positions.get(handle, None)
                    if handle_pos is None:
                        continue
                    # Find the associated drawer
                    for drawer in object_names:
                        if "drawer" in drawer:
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                target_drawer=drawer,
                                target_handle=handle,
                                target_pos=handle_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            print(f"  [Success] Pulled {handle} to open {drawer}.")
                            break
                except Exception as e:
                    print(f"  [Error] Could not pull {handle}: {e}")

        # 5. Try to sweep an object (if available)
        for obj in object_names:
            if "dice" in obj or "object" in obj:
                try:
                    print(f"[Exploration] Attempting to sweep {obj}...")
                    obj_pos = positions.get(obj, None)
                    if obj_pos is None:
                        continue
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        target_obj=obj,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"  [Success] Swept {obj}.")
                except Exception as e:
                    print(f"  [Error] Could not sweep {obj}: {e}")

        # 6. Try to use gripper skill (if available)
        try:
            print(f"[Exploration] Attempting to use gripper...")
            obs, reward, done = execute_gripper(
                env,
                task,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print(f"  [Success] Gripper skill executed.")
        except Exception as e:
            print(f"  [Error] Could not execute gripper skill: {e}")

        print("===== Exploration Phase Complete =====")
        print("If any errors above mention missing predicates (e.g., 'robot-free', 'hand-empty'), these are likely the missing predicates in the domain or initial state.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
