# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to explore the environment and check for missing predicates.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # This will help us determine if any predicate is missing or if an action fails due to a missing precondition.

        # Get all objects, drawers, and locations from positions
        objects = [k for k in positions.keys() if 'object' in k]
        drawers = [k for k in positions.keys() if 'drawer' in k]
        handles = [k for k in positions.keys() if 'handle' in k]
        locations = [k for k in positions.keys() if 'location' in k or 'table' in k or 'floor' in k]

        # Fallback: If no explicit locations, use all unique position values as locations
        if not locations:
            locations = list(set([v for v in positions.values()]))

        # Try to execute each skill and catch exceptions to infer missing predicates
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_go':
                    # Try moving between two locations if possible
                    if len(locations) >= 2:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"Executed {skill} from {from_loc} to {to_loc}")
                elif skill == 'execute_pick':
                    # Try picking up an object from a location
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"Executed {skill} on {obj} at {loc}")
                elif skill == 'execute_place':
                    # Try placing an object into a drawer at a location
                    if objects and drawers and locations:
                        obj = objects[0]
                        drawer = drawers[0]
                        loc = locations[0]
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"Executed {skill} for {obj} into {drawer} at {loc}")
                elif skill == 'execute_push':
                    # Try pushing (closing) a drawer at a location
                    if drawers and locations:
                        drawer = drawers[0]
                        loc = locations[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"Executed {skill} on {drawer} at {loc}")
                elif skill == 'execute_pull':
                    # Try pulling (opening) a drawer using a handle at a location
                    if drawers and handles and locations:
                        drawer = drawers[0]
                        handle = handles[0]
                        loc = locations[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"Executed {skill} on {drawer} with {handle} at {loc}")
                elif skill == 'execute_sweep':
                    # Try sweeping an object at a location
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"Executed {skill} on {obj} at {loc}")
                elif skill == 'execute_rotate':
                    # If rotate is available, try rotating an object (if such a function exists)
                    # Since no parameters are specified, we skip or call with dummy values
                    pass
                elif skill == 'execute_gripper':
                    # Try gripper action (if available)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"Executed {skill}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")

        # After exploration, print a message to indicate that missing predicates should be checked
        print("[Exploration] Exploration phase complete. Check logs for failed skills to infer missing predicates.")

        # === (Optional) Main Task Plan Execution ===
        # If an oracle plan is available, execute it step by step using the available skills.
        # For this template, we assume the exploration phase is the main focus due to missing predicate feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()