# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will perform an exploration using available skills to check and ensure the robot is in a 'free' state.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_gripper' and 'execute_sweep' to see if they affect the robot-free state,
        # and also check the effect of 'execute_pick' and 'execute_place' on the robot's state.

        # For demonstration, we will:
        # 1. Move the robot to a location (if needed)
        # 2. Attempt to pick an object (if available)
        # 3. Attempt to place the object (if possible)
        # 4. Use execute_gripper and execute_sweep to see if they affect robot-free
        # 5. Print out observations after each step to check for the robot-free state

        # --- Step 1: Get a list of objects and locations from positions ---
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)
        # Fallback: If no explicit locations, use all unique positions as locations
        if not location_names:
            location_names = list(set([v for k, v in positions.items()]))

        # --- Step 2: Try to pick an object from the floor ---
        picked_object = None
        pick_location = None
        for obj in object_names:
            # Try to find an object that is on the floor and reachable
            # For this example, we assume the first object is suitable
            picked_object = obj
            # Try to find its location
            pick_location = None
            for loc in location_names:
                if positions.get(obj) == positions.get(loc):
                    pick_location = loc
                    break
            if pick_location is None and location_names:
                pick_location = location_names[0]
            break

        # --- Step 3: Execute pick if possible ---
        if picked_object and pick_location:
            print(f"[Exploration] Attempting to pick {picked_object} at {pick_location}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=picked_object,
                    location=pick_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Pick executed.")
            except Exception as e:
                print(f"[Exploration] Pick failed: {e}")
        else:
            print("[Exploration] No suitable object/location found for pick.")

        # --- Step 4: Try to place the object in a drawer if possible ---
        placed = False
        if picked_object and drawer_names and pick_location:
            drawer = drawer_names[0]
            print(f"[Exploration] Attempting to place {picked_object} in {drawer} at {pick_location}")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj=picked_object,
                    drawer=drawer,
                    location=pick_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                placed = True
                print("[Exploration] Place executed.")
            except Exception as e:
                print(f"[Exploration] Place failed: {e}")

        # --- Step 5: Try to use execute_gripper and execute_sweep ---
        print("[Exploration] Attempting to execute_gripper (should set robot-free if not already).")
        try:
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=50,
                timeout=5.0
            )
            print("[Exploration] execute_gripper executed.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Attempting to execute_sweep (should not affect robot-free).")
        try:
            if picked_object and pick_location:
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj=picked_object,
                    location=pick_location,
                    max_steps=50,
                    timeout=5.0
                )
                print("[Exploration] execute_sweep executed.")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # --- Step 6: Print out the current observation to check for robot-free ---
        print("[Exploration] Checking for 'robot-free' in observation/state.")
        try:
            current_obs = task.get_observation()
            if hasattr(current_obs, 'state') and 'robot-free' in current_obs.state:
                print("[Exploration] 'robot-free' predicate is present in the state.")
            else:
                print("[Exploration] 'robot-free' predicate is NOT present in the state.")
        except Exception as e:
            print(f"[Exploration] Could not check for 'robot-free': {e}")

        # --- End of Exploration Phase ---

        # === (Optional) Continue with Oracle Plan Execution ===
        # At this point, you would continue with the actual oracle plan using the available skills,
        # now that you have explored and checked the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()