# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We need to explore the environment to ensure the robot can be at 'drawer-area' and that this predicate is recognized.

        # For this, we will use the available skills to move the robot to the 'drawer-area' and check the state.

        # Step 1: Determine current robot location and the 'drawer-area' location
        # Assume positions dict contains keys like 'robot', 'drawer-area', etc.
        try:
            robot_pos = positions['robot']
        except KeyError:
            print("[Exploration] Could not find 'robot' in positions. Exploration aborted.")
            return

        # Try to find a location key that matches 'drawer-area'
        drawer_area_key = None
        for key in positions:
            if 'drawer-area' in key:
                drawer_area_key = key
                break
        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in positions. Exploration aborted.")
            return
        drawer_area_pos = positions[drawer_area_key]

        # Step 2: Move robot to 'drawer-area' using the available skill
        # The skill is 'execute_go', which takes (from, to) locations
        # We need to know the names of the locations, not just their positions.
        # For this example, let's assume the keys in positions are the location names.

        # Find the robot's current location name
        robot_location_name = None
        for loc_name, pos in positions.items():
            if loc_name == 'robot':
                continue
            # If robot's position matches this location (within a small threshold), assign as current location
            if np.linalg.norm(np.array(robot_pos) - np.array(pos)) < 0.05:
                robot_location_name = loc_name
                break
        if robot_location_name is None:
            # If not found, just pick a default starting location (first non-robot, non-drawer-area)
            for loc_name in positions:
                if loc_name not in ['robot', drawer_area_key]:
                    robot_location_name = loc_name
                    break

        # Now, execute the move to 'drawer-area'
        print(f"[Exploration] Moving robot from '{robot_location_name}' to '{drawer_area_key}' to explore predicate (robot-at drawer-area).")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location_name,
                to_location=drawer_area_key
            )
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            return

        # Step 3: After moving, check if the predicate (robot-at drawer-area) is now true in the environment
        # This would typically be done by querying the environment state or observation
        # For demonstration, print out the current observation or state
        try:
            current_obs = task.get_observation()
            print("[Exploration] Current observation after moving to drawer-area:", current_obs)
        except Exception as e:
            print(f"[Exploration] Exception while getting observation: {e}")

        # Step 4: (Optional) Try to perform an action that requires (robot-at drawer-area) as a precondition
        # For example, try to open a drawer if such a skill is available and the drawer is at 'drawer-area'
        # This step is optional and depends on the actual environment and available skills

        print("[Exploration] Exploration phase complete. If (robot-at drawer-area) is now true, the missing predicate is confirmed.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()