# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding drawer_top_handle)
        # The plan failed due to a missing predicate, likely related to the lock state of the drawer.
        # According to the exploration domain, we can use execute_pull to discover lock-known(?obj).
        # We will attempt to pull the handle and observe if the lock-known predicate is discovered.

        # --- Identify relevant objects and locations ---
        # These names are placeholders; replace with actual names from your environment if needed.
        drawer_handle_name = 'drawer_top_handle'
        drawer_name = 'drawer_top'
        robot_location = None
        handle_location = None

        # Find the location of the handle
        for obj_name, pos in positions.items():
            if obj_name == drawer_handle_name:
                handle_location = pos
            # Optionally, set robot_location if available
            if 'robot' in obj_name:
                robot_location = pos

        if handle_location is None:
            print("[Exploration] Could not find handle location in object positions.")
            return

        # Move robot to the handle location if not already there
        try:
            # If robot_location is not at handle_location, move
            if robot_location is not None and robot_location != handle_location:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=handle_location
                )
                print(f"[Exploration] Robot moved to handle location: {handle_location}")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Try to pick the handle
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=drawer_handle_name,
                p=handle_location
            )
            print(f"[Exploration] Picked up handle: {drawer_handle_name}")
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Try to pull the handle to discover lock-known
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                d=drawer_name,
                h=drawer_handle_name,
                p=handle_location
            )
            print(f"[Exploration] Pulled handle to check lock state for: {drawer_name}")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")

        # At this point, the missing predicate (lock-known) should be discovered if it was missing.
        # The rest of the plan can proceed as normal.

        # === Main Task Plan ===
        # Example: Open the drawer, place an object, close the drawer

        # 1. Ensure robot is at the drawer location
        try:
            # If not already at drawer location, move
            if robot_location is not None and robot_location != handle_location:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=handle_location
                )
                print(f"[Task] Robot moved to drawer location: {handle_location}")
        except Exception as e:
            print(f"[Task] Error during execute_go: {e}")

        # 2. If not holding the handle, pick it up
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=drawer_handle_name,
                p=handle_location
            )
            print(f"[Task] Picked up handle: {drawer_handle_name}")
        except Exception as e:
            print(f"[Task] Error during execute_pick: {e}")

        # 3. Pull the handle to open the drawer
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                d=drawer_name,
                h=drawer_handle_name,
                p=handle_location
            )
            print(f"[Task] Pulled handle to open drawer: {drawer_name}")
        except Exception as e:
            print(f"[Task] Error during execute_pull: {e}")

        # 4. Pick up an object from the floor (example: 'object_1')
        object_to_pick = None
        object_location = None
        for obj_name, pos in positions.items():
            if obj_name.startswith('object_'):
                object_to_pick = obj_name
                object_location = pos
                break

        if object_to_pick is not None and object_location is not None:
            try:
                # Move to object location if needed
                if handle_location != object_location:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=handle_location,
                        to_location=object_location
                    )
                    print(f"[Task] Robot moved to object location: {object_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_to_pick,
                    p=object_location
                )
                print(f"[Task] Picked up object: {object_to_pick}")
            except Exception as e:
                print(f"[Task] Error during execute_pick for object: {e}")

            # Move back to drawer location
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=object_location,
                    to_location=handle_location
                )
                print(f"[Task] Robot moved back to drawer location: {handle_location}")
            except Exception as e:
                print(f"[Task] Error during execute_go to drawer: {e}")

            # Place the object in the drawer
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    o=object_to_pick,
                    d=drawer_name,
                    p=handle_location
                )
                print(f"[Task] Placed object {object_to_pick} in drawer {drawer_name}")
            except Exception as e:
                print(f"[Task] Error during execute_place: {e}")

        # 5. Push the drawer to close it
        try:
            obs, reward, done = execute_push(
                env,
                task,
                d=drawer_name,
                p=handle_location
            )
            print(f"[Task] Pushed drawer {drawer_name} to close it.")
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
