# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use available skills to interact with objects/drawers and observe the environment state.

        # For demonstration, we will:
        # 1. Try to open each drawer (execute_pull) and check if it is locked/unlocked.
        # 2. Try to pick up each object (execute_pick) and see if it is possible.
        # 3. Move the robot to each location (execute_go) and observe.

        # Get lists of objects, drawers, locations from positions or descriptions
        # (Assume positions provides keys like 'objects', 'drawers', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])

        # If not provided, try to infer from descriptions or obs
        if not objects or not drawers or not locations:
            # Fallback: try to parse from descriptions or obs
            # This is a placeholder; in practice, you would extract these from the environment
            objects = []
            drawers = []
            locations = []
            for k in positions.keys():
                if 'object' in k:
                    objects.append(k)
                if 'drawer' in k:
                    drawers.append(k)
                if 'location' in k:
                    locations.append(k)
            # If still empty, use some default names for demonstration
            if not objects:
                objects = ['object_1']
            if not drawers:
                drawers = ['drawer_1']
            if not locations:
                locations = ['location_1']

        # Assume robot starts at the first location
        current_location = locations[0]

        # 1. Try to move to each location (explore with execute_go)
        for loc in locations:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Moving from {current_location} to {loc}")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                current_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick up each object at the current location
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to pick up {obj} at {current_location}")
                obs, reward, done = execute_pick(env, task, object_name=obj, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 3. Try to open each drawer using its handle (execute_pull)
        for drawer in drawers:
            # Try to find the handle for the drawer
            handle_name = f"{drawer}_handle"
            try:
                print(f"[Exploration] Attempting to pick up handle {handle_name} at {current_location}")
                obs, reward, done = execute_pick(env, task, object_name=handle_name, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick up handle {handle_name}: {e}")
            try:
                print(f"[Exploration] Attempting to pull open {drawer} using handle {handle_name} at {current_location}")
                obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle_name, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to pull open {drawer}: {e}")

        # 4. Try to place an object in a drawer (execute_place)
        for obj in objects:
            for drawer in drawers:
                try:
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {current_location}")
                    obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer, location=current_location)
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")

        # 5. Try to push drawers closed (execute_push)
        for drawer in drawers:
            try:
                print(f"[Exploration] Attempting to push close {drawer} at {current_location}")
                obs, reward, done = execute_push(env, task, drawer_name=drawer, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to push close {drawer}: {e}")

        # 6. Try to sweep objects (execute_sweep)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at {current_location}")
                obs, reward, done = execute_sweep(env, task, object_name=obj, location=current_location)
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze the environment state and logs to determine which predicate is missing.
        # This may require human inspection or further automated analysis.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()