# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions for object names (update as per your environment):
        # - drawer1: the drawer to be explored
        # - handle1: the handle object attached to drawer1
        # - robot starts at some initial location, e.g., 'loc_robot_start'
        # - drawer1 is at location 'loc_drawer1'
        # - handle1 is at location 'loc_drawer1' (attached to drawer1)
        # - If your environment uses different names, update accordingly.

        # You may need to update these names based on your environment's object naming
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_start_loc = 'loc_robot_start'
        drawer_loc = 'loc_drawer1'

        # Try to get actual positions if available
        # If not, use the above names as logical references
        try:
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            robot_pos = positions.get('robot', None)
        except Exception as e:
            print("[Exploration] Could not retrieve positions from object_positions:", e)
            drawer_pos = None
            handle_pos = None
            robot_pos = None

        # Step 1: Move robot to drawer location if not already there
        try:
            # If robot position and drawer position are available, use them
            if robot_pos is not None and drawer_pos is not None:
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=drawer_pos)
            else:
                # Use logical names if positions are not available
                obs, reward, done = execute_go(env, task, from_location=robot_start_loc, to_location=drawer_loc)
            print("[Exploration] Robot moved to drawer location.")
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # Step 2: Try to pick the handle (if required for pulling)
        try:
            obs, reward, done = execute_pick(env, task, object_name=handle_name, location=drawer_loc)
            print("[Exploration] Picked handle:", handle_name)
        except Exception as e:
            print("[Exploration] Error during execute_pick (handle):", e)

        # Step 3: Try to pull the drawer to check its state (drawer-closed, drawer-open, drawer-locked, etc.)
        # This will help us determine if (drawer-closed drawer1) is missing or needs to be set
        try:
            obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_loc)
            print("[Exploration] Attempted to pull the drawer to check its state.")
        except Exception as e:
            print("[Exploration] Error during execute_pull:", e)

        # Step 4: If the drawer was not open, try to push it closed (to set drawer-closed predicate)
        try:
            obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=drawer_loc)
            print("[Exploration] Attempted to push the drawer closed (set drawer-closed).")
        except Exception as e:
            print("[Exploration] Error during execute_push:", e)

        # At this point, the exploration phase should have interacted with the drawer and
        # attempted to set or check the (drawer-closed drawer1) predicate.
        # You can now proceed with the rest of your oracle plan as needed.

        # === (Optional) Continue with Oracle Plan ===
        # Insert further plan steps here as required by your task.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
