# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover the missing predicate.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and drawers, and observe the environment.

        # Get all objects, drawers, and locations from positions
        # Assume positions is a dict: {name: (type, position)}
        # For this exploration, we will try to interact with all objects and drawers

        # Extract object, drawer, and location names
        object_names = []
        drawer_names = []
        handle_names = []
        location_names = set()
        for name, info in positions.items():
            if 'object' in name:
                object_names.append(name)
            elif 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            # Assume all objects have a position, which can be used as a location
            location_names.add(info['location'] if 'location' in info else name)
        location_names = list(location_names)

        # For robustness, if positions is not detailed, fall back to some default names
        if not object_names:
            object_names = ['object_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not location_names:
            location_names = ['location_1']

        # Assume robot starts at the first location
        robot_location = location_names[0]

        # 1. Try to pick up each object from the floor (execute_pick)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick up {obj} at {robot_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 2. Try to open each drawer (execute_pull)
        for drawer in drawer_names:
            # Try to find a handle for this drawer
            handle = None
            for h in handle_names:
                # If handle-of relationship is known, use it; else, just pick the first handle
                if drawer in h:
                    handle = h
                    break
            if not handle and handle_names:
                handle = handle_names[0]
            if not handle:
                continue
            try:
                print(f"[Exploration] Attempting to pick up handle {handle} at {robot_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle,
                    robot_location
                )
                print(f"[Exploration] Attempting to pull drawer {drawer} with handle {handle} at {robot_location}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pulled (opened) drawer {drawer}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pull drawer {drawer}: {e}")

        # 3. Try to place an object into a drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {robot_location}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        robot_location
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}.")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} into {drawer}: {e}")

        # 4. Try to close each drawer (execute_push)
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to push (close) drawer {drawer} at {robot_location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) drawer {drawer}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to push drawer {drawer}: {e}")

        # 5. Try to move to other locations (execute_go)
        for to_location in location_names:
            if to_location == robot_location:
                continue
            try:
                print(f"[Exploration] Attempting to move from {robot_location} to {to_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    robot_location,
                    to_location
                )
                if done:
                    print(f"[Exploration] Successfully moved to {to_location}.")
                    robot_location = to_location
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move to {to_location}: {e}")

        # 6. Try to sweep objects (execute_sweep)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at {robot_location}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper at {robot_location}")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()