# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to check for missing predicates.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # This will help us determine if any predicate is missing or if an action fails due to a missing precondition.

        # Get all objects, drawers, and locations from positions
        objects = [k for k in positions.keys() if 'object' in k]
        drawers = [k for k in positions.keys() if 'drawer' in k]
        handles = [k for k in positions.keys() if 'handle' in k]
        locations = [k for k in positions.keys() if 'location' in k or 'room' in k or 'table' in k]

        # Fallback: If no explicit locations, use all unique position values as locations
        if not locations:
            locations = list(set([v for v in positions.values()]))

        # Try to execute each skill and catch exceptions to infer missing predicates
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] execute_pick succeeded for {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] execute_place succeeded for {obj} in {drawer} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} in {drawer} at {loc}: {e}")

        for drawer in drawers:
            for handle in handles:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_pull on {drawer} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] execute_pull succeeded for {drawer} with {handle} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")

        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] execute_push succeeded for {drawer} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] execute_sweep succeeded for {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # Try moving between locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] execute_go succeeded from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # Try gripper action if available
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] execute_gripper succeeded")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, the focus is on exploration to find missing predicates as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()