# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment.

        # Get all objects, drawers, and locations from positions
        # Assume positions is a dict: {name: (type, position)}
        objects = []
        drawers = []
        handles = []
        locations = []
        for name, info in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'handle' in name:
                handles.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)

        # Fallback: If no explicit locations, use unique positions as locations
        if not locations:
            loc_set = set()
            for name, info in positions.items():
                pos = info[1] if isinstance(info, tuple) and len(info) > 1 else None
                if pos is not None:
                    loc_set.add(pos)
            locations = list(loc_set)

        # For this exploration, we will:
        # 1. Try to pick up each object on the floor using execute_pick
        # 2. Try to open each drawer using execute_pull (requires holding the handle)
        # 3. Try to place an object in a drawer using execute_place
        # 4. Try to move between locations using execute_go

        # We will log any failures to infer missing predicates

        exploration_log = []

        # 1. Try to pick up each object
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc} using execute_pick")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    exploration_log.append(f"Picked {obj} at {loc}")
                    if done:
                        print(f"[Exploration] Task ended after picking {obj}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")
                    exploration_log.append(f"Failed to pick {obj} at {loc}: {e}")

        # 2. Try to open each drawer (requires holding the handle)
        for drawer in drawers:
            for handle in handles:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to pick handle {handle} at {loc} for drawer {drawer}")
                        obs, reward, done = execute_pick(env, task, handle, loc)
                        print(f"[Exploration] Attempting to pull drawer {drawer} with handle {handle} at {loc} using execute_pull")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        exploration_log.append(f"Pulled drawer {drawer} with handle {handle} at {loc}")
                        if done:
                            print(f"[Exploration] Task ended after pulling drawer {drawer}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Failed to pull drawer {drawer} with handle {handle} at {loc}: {e}")
                        exploration_log.append(f"Failed to pull drawer {drawer} with handle {handle} at {loc}: {e}")

        # 3. Try to place each object in each drawer at each location
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc} using execute_place")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        exploration_log.append(f"Placed {obj} in {drawer} at {loc}")
                        if done:
                            print(f"[Exploration] Task ended after placing {obj}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                        exploration_log.append(f"Failed to place {obj} in {drawer} at {loc}: {e}")

        # 4. Try to move between all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Attempting to move from {from_loc} to {to_loc} using execute_go")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    exploration_log.append(f"Moved from {from_loc} to {to_loc}")
                    if done:
                        print(f"[Exploration] Task ended after moving to {to_loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")
                    exploration_log.append(f"Failed to move from {from_loc} to {to_loc}: {e}")

        # 5. Try to push (close) each drawer at each location
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to push (close) drawer {drawer} at {loc} using execute_push")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    exploration_log.append(f"Pushed (closed) drawer {drawer} at {loc}")
                    if done:
                        print(f"[Exploration] Task ended after pushing drawer {drawer}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to push (close) drawer {drawer} at {loc}: {e}")
                    exploration_log.append(f"Failed to push (close) drawer {drawer} at {loc}: {e}")

        # 6. Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc} using execute_sweep")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    exploration_log.append(f"Swept {obj} at {loc}")
                    if done:
                        print(f"[Exploration] Task ended after sweeping {obj}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
                    exploration_log.append(f"Failed to sweep {obj} at {loc}: {e}")

        # 7. Try to use gripper (if applicable)
        try:
            print(f"[Exploration] Attempting to use gripper using execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_log.append(f"Used gripper")
            if done:
                print(f"[Exploration] Task ended after using gripper!")
                return
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")
            exploration_log.append(f"Failed to use gripper: {e}")

        # === Exploration Summary ===
        print("===== Exploration Log =====")
        for entry in exploration_log:
            print(entry)
        print("===== End of Exploration =====")

        # TODO: After exploration, analyze the log to infer which predicate is missing (e.g., lock-known, identified, etc.)
        # This can be done by looking for consistent failures that suggest a missing precondition.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()