# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use available skills to explore the environment and infer missing predicates.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use these skills in a safe exploratory sequence to trigger errors or discover missing predicates.

        # 1. Try to move the robot to all known locations
        # 2. Try to pick up all objects on the floor
        # 3. Try to open and close all drawers (if any)
        # 4. Try to place objects in drawers (if possible)
        # 5. Log any errors or unexpected failures

        # For demonstration, we assume positions is a dict like:
        # {'robot': (x, y, z), 'object_1': (x, y, z), 'drawer_1': (x, y, z), ...}
        # and that we have lists of object and location names

        # Extract object and location names from positions
        object_names = []
        drawer_names = []
        location_names = []
        handle_names = []

        for name in positions:
            if name.startswith('object') or name.startswith('obj'):
                object_names.append(name)
            elif name.startswith('drawer'):
                drawer_names.append(name)
            elif name.startswith('handle'):
                handle_names.append(name)
            elif name.startswith('loc') or name.startswith('room') or name.startswith('table'):
                location_names.append(name)

        # If no explicit locations, use all unique positions as locations
        if not location_names:
            location_names = list(set([str(pos) for pos in positions.values()]))

        # 1. Try to move the robot to all known locations
        print("[Exploration] Moving robot to all known locations...")
        last_location = None
        try:
            for loc in location_names:
                if last_location is not None and loc != last_location:
                    print(f"[Exploration] execute_go: {last_location} -> {loc}")
                    try:
                        obs, reward, done = execute_go(env, task, last_location, loc)
                    except Exception as e:
                        print(f"[Exploration][Error] execute_go({last_location}, {loc}): {e}")
                last_location = loc
        except Exception as e:
            print(f"[Exploration][Error] Moving to locations: {e}")

        # 2. Try to pick up all objects on the floor at each location
        print("[Exploration] Attempting to pick up all objects at all locations...")
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_pick: {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration][Error] execute_pick({obj}, {loc}): {e}")

        # 3. Try to open and close all drawers using handles (if any)
        print("[Exploration] Attempting to open and close all drawers...")
        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] execute_pull: {drawer} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                    except Exception as e:
                        print(f"[Exploration][Error] execute_pull({drawer}, {handle}, {loc}): {e}")
            # Try to close the drawer
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_push: {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration][Error] execute_push({drawer}, {loc}): {e}")

        # 4. Try to place all objects in all drawers at all locations
        print("[Exploration] Attempting to place all objects in all drawers at all locations...")
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] execute_place: {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration][Error] execute_place({obj}, {drawer}, {loc}): {e}")

        # 5. Try other available skills for completeness
        print("[Exploration] Attempting to sweep all objects at all locations...")
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] execute_sweep: {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration][Error] execute_sweep({obj}, {loc}): {e}")

        print("[Exploration] Attempting to use gripper skill...")
        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration][Error] execute_gripper(): {e}")

        print("[Exploration] Exploration phase complete. Check above for any errors or missing predicate hints.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()