# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with all objects and drawers,
        # and log any unexpected failures or state changes to help identify missing predicates.

        # Get all objects and drawers from the environment
        # (Assume positions dict contains keys for all objects and drawers)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'pos' in name]

        # If locations are not explicitly named, use unique positions as locations
        if not location_names:
            location_names = list(set([tuple(pos) for pos in positions.values()]))

        # Try to move to each location and interact with objects/drawers
        for loc_name in location_names:
            try:
                # Move to the location using execute_go if possible
                # We need to know the current location; assume the robot starts at the first location
                # For the first iteration, set current_location
                if 'current_location' not in locals():
                    current_location = loc_name
                if current_location != loc_name:
                    print(f"[Exploration] Moving from {current_location} to {loc_name}")
                    try:
                        obs, reward, done = execute_go(env, task, current_location, loc_name)
                        current_location = loc_name
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")
                        continue

                # Try to pick up any object at this location
                for obj_name in object_names:
                    try:
                        obj_pos = positions[obj_name]
                        # If the object is at this location (by name or position), try to pick
                        if obj_pos == positions.get(loc_name, obj_pos):
                            print(f"[Exploration] Attempting to pick {obj_name} at {loc_name}")
                            obs, reward, done = execute_pick(env, task, obj_name, loc_name)
                            if done:
                                print(f"[Exploration] Picked {obj_name} at {loc_name}")
                                # Place it back if possible
                                for drawer_name in drawer_names:
                                    try:
                                        print(f"[Exploration] Attempting to place {obj_name} in {drawer_name} at {loc_name}")
                                        obs, reward, done = execute_place(env, task, obj_name, drawer_name, loc_name)
                                        if done:
                                            print(f"[Exploration] Placed {obj_name} in {drawer_name}")
                                            break
                                    except Exception as e:
                                        print(f"[Exploration] execute_place failed: {e}")
                                break
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed: {e}")

                # Try to interact with drawers: pull (open), push (close)
                for drawer_name in drawer_names:
                    try:
                        # Find handle for this drawer
                        handle_for_drawer = None
                        for h in handle_names:
                            # Assume handle-of relation is encoded in the name or positions
                            if drawer_name in h:
                                handle_for_drawer = h
                                break
                        if handle_for_drawer:
                            # Try to pick the handle
                            print(f"[Exploration] Attempting to pick handle {handle_for_drawer} at {loc_name}")
                            try:
                                obs, reward, done = execute_pick(env, task, handle_for_drawer, loc_name)
                                if done:
                                    print(f"[Exploration] Picked handle {handle_for_drawer}")
                                    # Try to pull (open) the drawer
                                    print(f"[Exploration] Attempting to pull {drawer_name} with handle {handle_for_drawer} at {loc_name}")
                                    try:
                                        obs, reward, done = execute_pull(env, task, drawer_name, handle_for_drawer, loc_name)
                                        if done:
                                            print(f"[Exploration] Pulled (opened) {drawer_name}")
                                            # Try to push (close) the drawer
                                            print(f"[Exploration] Attempting to push (close) {drawer_name} at {loc_name}")
                                            try:
                                                obs, reward, done = execute_push(env, task, drawer_name, loc_name)
                                                if done:
                                                    print(f"[Exploration] Pushed (closed) {drawer_name}")
                                            except Exception as e:
                                                print(f"[Exploration] execute_push failed: {e}")
                                    except Exception as e:
                                        print(f"[Exploration] execute_pull failed: {e}")
                            except Exception as e:
                                print(f"[Exploration] execute_pick (handle) failed: {e}")
                    except Exception as e:
                        print(f"[Exploration] Drawer interaction failed: {e}")

                # Try to sweep any object at this location
                for obj_name in object_names:
                    try:
                        obj_pos = positions[obj_name]
                        if obj_pos == positions.get(loc_name, obj_pos):
                            print(f"[Exploration] Attempting to sweep {obj_name} at {loc_name}")
                            obs, reward, done = execute_sweep(env, task, obj_name, loc_name)
                            if done:
                                print(f"[Exploration] Swept {obj_name} at {loc_name}")
                    except Exception as e:
                        print(f"[Exploration] execute_sweep failed: {e}")

                # Try to use gripper (if applicable)
                try:
                    print(f"[Exploration] Attempting to use gripper at {loc_name}")
                    obs, reward, done = execute_gripper(env, task)
                    if done:
                        print(f"[Exploration] Used gripper at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_gripper failed: {e}")

            except Exception as e:
                print(f"[Exploration] Exploration at {loc_name} failed: {e}")

        print("===== Exploration Phase Complete =====")
        print("If any action failed due to a missing predicate, check the logs above for clues about the missing predicate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()