# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate/State ===
        # The feedback indicates we must check the state of the drawer before placing an object.
        # We need to check if the drawer is empty, partially-filled, or full before placing.
        # We'll use the available skills and the positions/state info to do this.

        # Example: Let's assume we have the following objects and drawers
        # (You should replace these with actual names from your environment)
        # For demonstration, we use generic names:
        object_to_store = None
        open_drawer = None
        drawer_handle = None
        robot_location = None
        drawer_location = None

        # Try to find an open drawer and an object to store
        for name, info in positions.items():
            if 'drawer' in name and info.get('state', '') == 'open':
                open_drawer = name
                drawer_location = info.get('location', None)
            if 'object' in name and info.get('on_floor', False):
                object_to_store = name
            if 'handle' in name and info.get('drawer', None) == open_drawer:
                drawer_handle = name
            if info.get('robot', False):
                robot_location = info.get('location', None)

        if open_drawer is None or object_to_store is None or drawer_location is None:
            print("[Exploration] Could not find open drawer or object to store. Exiting.")
            return

        # Check drawer state before placing
        drawer_state = positions[open_drawer].get('drawer_state', None)
        drawer_contents = positions[open_drawer].get('contents', [])
        drawer_capacity = positions[open_drawer].get('capacity', 1)

        # 1. If drawer is empty, proceed to place
        # 2. If drawer is partially-filled or full, check if object matches or handle conflict
        can_place = False
        if drawer_state == 'empty' or (drawer_contents == []):
            can_place = True
        elif drawer_state == 'partially-filled':
            if object_to_store not in drawer_contents and len(drawer_contents) < drawer_capacity:
                can_place = True
            else:
                print("[Exploration] Drawer already contains this object or is full.")
        elif drawer_state == 'full':
            print("[Exploration] Drawer is full. Cannot place object.")
        else:
            print("[Exploration] Unknown drawer state. Cannot proceed.")
        
        # If not able to place, handle conflict (e.g., skip or remove object)
        if not can_place:
            print("[Exploration] Handling conflict: cannot place object in drawer.")
            # Here, you could implement logic to remove an object or choose another drawer
            # For now, we exit
            return

        # === Task Plan Execution ===
        # 1. Move robot to object location if not already there
        object_location = positions[object_to_store].get('location', None)
        if robot_location != object_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {object_location}")
                obs, reward, done = execute_go(env, task, robot_location, object_location)
                robot_location = object_location
            except Exception as e:
                print(f"[Error] Failed to move robot: {e}")
                return

        # 2. Pick up the object
        try:
            print(f"[Task] Picking up object: {object_to_store} at {object_location}")
            obs, reward, done = execute_pick(env, task, object_to_store, object_location)
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 3. Move robot to drawer location if not already there
        if robot_location != drawer_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Error] Failed to move robot: {e}")
                return

        # 4. Place the object in the drawer
        try:
            print(f"[Task] Placing object: {object_to_store} in drawer: {open_drawer} at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_to_store, open_drawer, drawer_location)
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")
            return

        # 5. Optionally, close the drawer if required
        try:
            print(f"[Task] Closing drawer: {open_drawer} at {drawer_location}")
            obs, reward, done = execute_push(env, task, open_drawer, drawer_location)
        except Exception as e:
            print(f"[Error] Failed to close drawer: {e}")
            # Not critical, so continue

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()