# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to check for missing predicates.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will try to use execute_pull (which in the exploration domain is used to discover lock-known)
        # and execute_pick (which in the exploration domain is used to discover weight-known/durability-known).
        # Since the main domain does not have these predicates, we will use the skills to interact and observe.

        # For demonstration, we will attempt to pull on all handles to see if any lock-related predicate is missing.
        # We will also attempt to pick up all objects to see if any weight/durability predicate is missing.

        # Get all objects and drawers from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Try to execute pull on all handles (to check for lock-known predicate)
        for handle in handle_names:
            # Find which drawer this handle belongs to (if possible)
            for drawer in drawer_names:
                try:
                    # Try to pull the drawer using the handle
                    print(f"[Exploration] Attempting to pull handle {handle} of drawer {drawer}")
                    # Find a location near the drawer
                    drawer_pos = positions[drawer]
                    # Find the closest location
                    if location_names:
                        loc = location_names[0]
                    else:
                        loc = None
                    # Move to the location if possible
                    if loc is not None:
                        try:
                            obs, reward, done = execute_go(env, task, from_location=None, to_location=loc)
                        except Exception as e:
                            print(f"[Exploration] Could not move to location {loc}: {e}")
                    # Pick the handle first (if required)
                    try:
                        obs, reward, done = execute_pick(env, task, obj=handle, location=loc)
                    except Exception as e:
                        print(f"[Exploration] Could not pick handle {handle}: {e}")
                    # Now try to pull
                    try:
                        obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=loc)
                        print(f"[Exploration] Pulled handle {handle} of drawer {drawer}")
                    except Exception as e:
                        print(f"[Exploration] Could not pull handle {handle} of drawer {drawer}: {e}")
                except Exception as e:
                    print(f"[Exploration] Error in pulling handle {handle} of drawer {drawer}: {e}")

        # Try to pick up all objects (to check for weight-known/durability-known predicates)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to pick object {obj} at location {loc}")
                    obs, reward, done = execute_pick(env, task, obj=obj, location=loc)
                    print(f"[Exploration] Picked object {obj} at location {loc}")
                except Exception as e:
                    print(f"[Exploration] Could not pick object {obj} at location {loc}: {e}")

        # Try to place objects in drawers (to check for drawer-full, drawer-empty, etc.)
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place object {obj} in drawer {drawer} at location {loc}")
                        obs, reward, done = execute_place(env, task, obj=obj, drawer=drawer, location=loc)
                        print(f"[Exploration] Placed object {obj} in drawer {drawer} at location {loc}")
                    except Exception as e:
                        print(f"[Exploration] Could not place object {obj} in drawer {drawer} at location {loc}: {e}")

        # Try to push all drawers (to check for drawer-open/drawer-closed predicates)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to push drawer {drawer} at location {loc}")
                    obs, reward, done = execute_push(env, task, drawer=drawer, location=loc)
                    print(f"[Exploration] Pushed drawer {drawer} at location {loc}")
                except Exception as e:
                    print(f"[Exploration] Could not push drawer {drawer} at location {loc}: {e}")

        # Try to sweep all objects (to check for on-floor predicate)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to sweep object {obj} at location {loc}")
                    obs, reward, done = execute_sweep(env, task, obj=obj, location=loc)
                    print(f"[Exploration] Swept object {obj} at location {loc}")
                except Exception as e:
                    print(f"[Exploration] Could not sweep object {obj} at location {loc}: {e}")

        # Try to use gripper (to check for robot-free/hand-empty predicates)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            print(f"[Exploration] Used gripper")
        except Exception as e:
            print(f"[Exploration] Could not use gripper: {e}")

        # Try to move between all locations (to check for robot-at predicate)
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_location=from_loc, to_location=to_loc)
                        print(f"[Exploration] Moved from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # Since the feedback is about missing predicates, the main focus is on exploration.
        # If the missing predicate is discovered and fixed, you can proceed to execute the main plan here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()