# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # This phase is based on the exploration knowledge provided.

        # For demonstration, we will attempt to use the available skills to interact with objects and drawers,
        # and print out the results to help identify what predicate might be missing.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers in the environment
        # The actual object and drawer names should be present in the positions dictionary
        # For this example, we will use generic names; in practice, replace with actual names

        # Example object and drawer names (replace with actual names from your environment)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # If no explicit location names, use all unique positions as locations
        if not location_names:
            location_names = list(set([positions[name] for name in positions]))

        # Assume robot starts at some location
        robot_location = None
        for name in location_names:
            # If the robot is at this location, set as starting location
            if 'robot' in name or 'start' in name:
                robot_location = name
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        # Exploration: Try to pick each object from the floor
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick {obj} at {robot_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj}")
                else:
                    print(f"[Exploration] Could not pick {obj} (possibly missing predicate or precondition)")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj}: {e}")

        # Exploration: Try to open each drawer using execute_pull
        for drawer in drawer_names:
            # Find handle for this drawer
            handle_name = None
            for obj in object_names:
                if f"handle_of_{drawer}" in obj or f"handle_{drawer}" in obj or "handle" in obj:
                    handle_name = obj
                    break
            if handle_name is None and object_names:
                handle_name = object_names[0]  # fallback to any object

            try:
                print(f"[Exploration] Attempting to pull {drawer} with handle {handle_name} at {robot_location}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_name,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pulled (opened) {drawer}")
                else:
                    print(f"[Exploration] Could not pull {drawer} (possibly missing predicate or precondition)")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull for {drawer}: {e}")

        # Exploration: Try to push (close) each drawer
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to push (close) {drawer} at {robot_location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) {drawer}")
                else:
                    print(f"[Exploration] Could not push {drawer} (possibly missing predicate or precondition)")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")

        # Exploration: Try to place each object into each drawer
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {robot_location}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        robot_location
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer} (possibly missing predicate or precondition)")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place for {obj} into {drawer}: {e}")

        # Exploration: Try to move (go) between locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Attempting to go from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_loc,
                        to_loc
                    )
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                    else:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc} (possibly missing predicate or precondition)")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go from {from_loc} to {to_loc}: {e}")

        # Exploration: Try to sweep each object
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at {robot_location}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}")
                else:
                    print(f"[Exploration] Could not sweep {obj} (possibly missing predicate or precondition)")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep for {obj}: {e}")

        # Exploration: Try to use gripper
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper")
            else:
                print(f"[Exploration] Could not use gripper (possibly missing predicate or precondition)")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Review the above logs to identify which predicates or preconditions are missing or failing.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem files accordingly and rerun the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()