# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We need to explore the environment to ensure the robot can reach and recognize the 'drawer-area' location.

        # 1. Find the current robot location and the 'drawer-area' location
        robot_location = None
        drawer_area_location = None

        # Try to infer robot and drawer-area from positions dictionary
        for key in positions:
            if 'robot' in key or 'Robot' in key:
                robot_location = positions[key]
            if 'drawer-area' in key or 'drawer' in key:
                drawer_area_location = positions[key]
        if robot_location is None:
            # Fallback: try to get from task or env
            try:
                robot_location = task.get_robot_position()
            except Exception:
                robot_location = (0, 0, 0)  # Default/fallback
        if drawer_area_location is None:
            # Fallback: try to get from task or env
            try:
                drawer_area_location = task.get_drawer_area_position()
            except Exception:
                drawer_area_location = (1, 0, 0)  # Default/fallback

        print(f"[Exploration] Robot initial location: {robot_location}")
        print(f"[Exploration] Drawer area location: {drawer_area_location}")

        # 2. Use the available skill 'execute_go' to move the robot to the drawer-area
        #    This will help us check if the predicate (robot-at drawer-area) is supported in the environment

        # The execute_go skill may require symbolic names, not coordinates.
        # Try to extract symbolic location names from the positions dict
        robot_loc_name = None
        drawer_area_name = None
        for name in positions:
            if 'robot' in name or 'Robot' in name:
                robot_loc_name = name
            if 'drawer-area' in name:
                drawer_area_name = name
            elif 'drawer' in name and drawer_area_name is None:
                drawer_area_name = name  # fallback

        # If not found, use default names
        if robot_loc_name is None:
            robot_loc_name = 'start_area'
        if drawer_area_name is None:
            drawer_area_name = 'drawer-area'

        print(f"[Exploration] Moving robot from '{robot_loc_name}' to '{drawer_area_name}' using execute_go.")

        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_loc_name,
                to_location=drawer_area_name
            )
            print(f"[Exploration] Robot moved to '{drawer_area_name}'.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # 3. After moving, check if the predicate (robot-at drawer-area) is now true in the environment
        #    This can be done by querying the environment state or by checking the observation

        # Try to check if robot is at drawer-area
        robot_at_drawer_area = False
        try:
            # If the observation contains location info, check it
            if hasattr(task, 'get_robot_location_name'):
                current_loc = task.get_robot_location_name()
                if current_loc == drawer_area_name:
                    robot_at_drawer_area = True
            elif hasattr(task, 'get_robot_position'):
                current_pos = task.get_robot_position()
                if np.allclose(current_pos, drawer_area_location, atol=0.05):
                    robot_at_drawer_area = True
            else:
                # Fallback: check obs dict
                if 'robot_location' in obs and obs['robot_location'] == drawer_area_name:
                    robot_at_drawer_area = True
        except Exception as e:
            print(f"[Exploration] Could not verify robot location: {e}")

        if robot_at_drawer_area:
            print("[Exploration] Predicate (robot-at drawer-area) is TRUE after execute_go.")
        else:
            print("[Exploration] Predicate (robot-at drawer-area) is still FALSE after execute_go.")

        # 4. If the predicate is missing, this exploration step helps identify it for future planning.
        #    You can now proceed with the rest of your oracle plan, knowing that (robot-at drawer-area) is a key predicate.

        # === (Continue with Oracle Plan as needed) ===
        # For this exploration, we stop here after verifying the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()