# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will use the available exploration skills to probe the environment and
        # try to discover which predicate is missing (e.g., lock-known, weight-known, etc.).
        # This phase uses only the available skills and does not define new ones.

        # For demonstration, we will attempt to use the available skills to interact
        # with all objects and drawers, and print out any issues encountered.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Attempt to explore all objects and drawers in the environment
        # The actual object and drawer names/types should be obtained from the environment
        # For this example, we assume positions contains keys for objects and drawers

        # Extract object and drawer names from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'area' in name]

        # If no explicit locations, use all unique positions as locations
        if not location_names:
            location_names = list(set([positions[name] for name in positions]))

        # 1. Try to pick up each object from the floor (execute_pick)
        for obj in object_names:
            try:
                # Find the location of the object
                obj_pos = positions[obj]
                # Find the closest location name (if available)
                obj_loc = None
                for loc in location_names:
                    if isinstance(loc, str) and positions.get(loc) == obj_pos:
                        obj_loc = loc
                        break
                if obj_loc is None:
                    obj_loc = obj_pos  # fallback to position tuple

                print(f"[Exploration] Attempting to pick up {obj} at {obj_loc}")
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully picked up {obj}")
                else:
                    print(f"[Exploration] Could not pick up {obj}")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj}: {e}")

        # 2. Try to open each drawer (execute_pull)
        for drawer in drawer_names:
            try:
                # Find handle object for the drawer
                handle_obj = None
                for obj in object_names:
                    # Assume handle objects are named as 'handle_of_drawerX'
                    if f"handle_of_{drawer}" in obj or "handle" in obj:
                        handle_obj = obj
                        break
                if handle_obj is None and object_names:
                    handle_obj = object_names[0]  # fallback to first object

                # Find the location of the drawer
                drawer_pos = positions[drawer]
                drawer_loc = None
                for loc in location_names:
                    if isinstance(loc, str) and positions.get(loc) == drawer_pos:
                        drawer_loc = loc
                        break
                if drawer_loc is None:
                    drawer_loc = drawer_pos

                print(f"[Exploration] Attempting to open {drawer} with handle {handle_obj} at {drawer_loc}")
                # First, pick the handle if not already holding
                try:
                    obs, reward, done = execute_pick(env, task, handle_obj, drawer_loc)
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pick for handle {handle_obj}: {e}")
                # Then, try to pull the drawer
                obs, reward, done = execute_pull(env, task, drawer, handle_obj, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully opened {drawer}")
                else:
                    print(f"[Exploration] Could not open {drawer}")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull for {drawer}: {e}")

        # 3. Try to place objects into drawers (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    # Find the location of the drawer
                    drawer_pos = positions[drawer]
                    drawer_loc = None
                    for loc in location_names:
                        if isinstance(loc, str) and positions.get(loc) == drawer_pos:
                            drawer_loc = loc
                            break
                    if drawer_loc is None:
                        drawer_loc = drawer_pos

                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {drawer_loc}")
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_loc)
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer}")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place for {obj} into {drawer}: {e}")

        # 4. Try to push drawers closed (execute_push)
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                drawer_loc = None
                for loc in location_names:
                    if isinstance(loc, str) and positions.get(loc) == drawer_pos:
                        drawer_loc = loc
                        break
                if drawer_loc is None:
                    drawer_loc = drawer_pos

                print(f"[Exploration] Attempting to push {drawer} closed at {drawer_loc}")
                obs, reward, done = execute_push(env, task, drawer, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully pushed {drawer} closed")
                else:
                    print(f"[Exploration] Could not push {drawer} closed")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")

        # 5. Try to move between locations (execute_go)
        if len(location_names) >= 2:
            for i in range(len(location_names) - 1):
                from_loc = location_names[i]
                to_loc = location_names[i + 1]
                try:
                    print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                    else:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc}")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go from {from_loc} to {to_loc}: {e}")

        # 6. Try to sweep objects (execute_sweep)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                obj_loc = None
                for loc in location_names:
                    if isinstance(loc, str) and positions.get(loc) == obj_pos:
                        obj_loc = loc
                        break
                if obj_loc is None:
                    obj_loc = obj_pos

                print(f"[Exploration] Attempting to sweep {obj} at {obj_loc}")
                obs, reward, done = execute_sweep(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully swept {obj}")
                else:
                    print(f"[Exploration] Could not sweep {obj}")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep for {obj}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
            else:
                print(f"[Exploration] Could not use gripper")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above for any missing predicates or failed actions.")

        # === End of Exploration ===

        # At this point, based on the exploration output, you should be able to identify
        # which predicate is missing (e.g., lock-known, weight-known, etc.) and update
        # your domain/problem files accordingly for future planning.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()