# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----------- EXPLORATION PHASE TO FIND MISSING PREDICATE -------------
        # Based on feedback, perform exploration with predefined skills to discover environment predicates.
        # The goal is to uncover what might be missing (e.g., lock/durability/weight/identification, etc).

        # Get the list of objects and locations from positions or from descriptions (if available)
        objects = [k for k, v in positions.items() if v.get("type", "").lower() == "object"]
        drawers = [k for k, v in positions.items() if v.get("type", "").lower() == "drawer"]
        locations = [v['location'] for k, v in positions.items() if 'location' in v]
        robot_location = None
        for k, v in positions.items():
            if v.get("type", "").lower() == "robot":
                robot_location = v.get("location", None)
                break

        # --- Safety checks for discovered elements ---
        if not objects:
            print("[Exploration] No objects found in positions dictionary.")
        if not drawers:
            print("[Exploration] No drawers found in positions dictionary.")
        if not locations:
            print("[Exploration] No locations found in positions dictionary.")
        if robot_location is None:
            print("[Exploration] Robot location not found in positions dictionary.")

        # 1. Exploration Skills: try available commands for information gathering
        for obj in objects:
            obj_info = positions.get(obj, {})
            obj_location = obj_info.get('location', None)
            if obj_location is None or robot_location is None:
                continue

            # Try to pick up the object to possibly determine its weight/durability/pickability
            try:
                # Only proceed if hand and robot are free at the location
                res = execute_pick(env, task, obj, obj_location)
                print(f"[Exploration] Tried picking {obj} at {obj_location}: {res}")
            except Exception as e:
                print(f"[Exploration] Error during execute_pick({obj}): {e}")

            # Try placing if holding and there is a drawer
            for drawer in drawers:
                drawer_info = positions.get(drawer, {})
                drawer_location = drawer_info.get('location', None)
                if drawer_location is not None and obj_location is not None:
                    try:
                        res = execute_place(env, task, obj, drawer, obj_location)
                        print(f"[Exploration] Tried placing {obj} in {drawer}: {res}")
                    except Exception as e:
                        print(f"[Exploration] Error during execute_place({obj},{drawer}): {e}")

        # 2. Try manipulation on drawers to test for possible locking/locking knowledge
        for drawer in drawers:
            drawer_info = positions.get(drawer, {})
            drawer_location = drawer_info.get('location', None)
            handle = None
            # Find associated handle, if specified
            for obj, val in positions.items():
                if val.get("type", "").lower() == "handle" and val.get('drawer') == drawer:
                    handle = obj
                    break
            # Try to pull the drawer (might reveal lock knowledge, or missing predicate)
            if handle is not None and drawer_location is not None:
                try:
                    # Assumes you need to hold the handle first
                    res = execute_pick(env, task, handle, drawer_location)
                    print(f"[Exploration] Tried picking handle {handle} at {drawer_location}: {res}")
                except Exception as e:
                    print(f"[Exploration] Error during pick handle({handle}): {e}")
                try:
                    res = execute_pull(env, task, drawer, handle, drawer_location)
                    print(f"[Exploration] Tried pulling {drawer} (handle {handle}): {res}")
                except Exception as e:
                    print(f"[Exploration] Error during execute_pull({drawer}): {e}")

        # 3. Try pushing all drawers (test closed→open state or missing knowledge predicate)
        for drawer in drawers:
            drawer_info = positions.get(drawer, {})
            drawer_location = drawer_info.get('location', None)
            if drawer_location is not None:
                try:
                    res = execute_push(env, task, drawer, drawer_location)
                    print(f"[Exploration] Tried pushing {drawer} at {drawer_location}: {res}")
                except Exception as e:
                    print(f"[Exploration] Error during execute_push({drawer}): {e}")

        # 4. Try go and sweep actions at all locations
        for loc_from in locations:
            for loc_to in locations:
                if loc_from == loc_to:
                    continue
                try:
                    res = execute_go(env, task, loc_from, loc_to)
                    print(f"[Exploration] Tried moving from {loc_from} to {loc_to}: {res}")
                except Exception as e:
                    print(f"[Exploration] Error during execute_go({loc_from}, {loc_to}): {e}")
        for obj in objects:
            obj_info = positions.get(obj, {})
            obj_location = obj_info.get('location', None)
            if obj_location is not None:
                try:
                    res = execute_sweep(env, task, obj, obj_location)
                    print(f"[Exploration] Tried sweeping {obj}: {res}")
                except Exception as e:
                    print(f"[Exploration] Error during execute_sweep({obj}): {e}")

        # 5. Try gripper operation (test for missing predicates or error feedback)
        try:
            res = execute_gripper(env, task)
            print(f"[Exploration] Tried execute_gripper: {res}")
        except Exception as e:
            print(f"[Exploration] Error during execute_gripper: {e}")

        # =================== END OF EXPLORATION ==========================
        # (In realistic code, you would analyze results to deduce missing predicates.)

        # === Domain-specific main logic (oracle plan) would go here ===
        # For now, since main plan is blocked due to missing predicate, we finish after exploration.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
