# run_skeleton_task.py (Complete Version with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided primitives and skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now including exploration to identify missing predicates like (drawer-closed drawer1).'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Optional: Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # This assumes get_object_positions() returns {object_name: position_tuple, ...}
        positions = get_object_positions()
        # For this scenario, assume object/drawer/handle naming conventions:
        # - drawers: e.g. 'drawer1'
        # - handles: e.g. 'handle1'
        # - locations: e.g. 'loc1', etc.
        #
        # If your object naming differs, adapt accordingly.
        #
        # For our exploration, we focus on discovering the predicate (drawer-closed drawer1).
        # So we must attempt actions whose effects depend on this predicate or try to trigger an effect with and without it.

        # === EXPLORATION PHASE ===
        print("[Exploration] Checking the status of drawer1 to determine if the (drawer-closed drawer1) predicate exists.")

        # Attempt to open (pull) 'drawer1' using its handle -- requires knowing handle and location.
        # For this, let's define names for drawer, handle, and robot location.
        drawer = 'drawer1'
        handle = 'handle1'
        robot_location = 'loc1'  # Assuming the initial location is 'loc1'
        drawer_location = 'loc_drawer1'  # If you have a mapping, use it

        # Try to go to the drawer location (if needed)
        try:
            # If not already at drawer location, move
            # You may need to determine the current and target locations; example
            current_loc = robot_location
            target_loc = drawer_location if drawer_location in positions else current_loc

            if current_loc != target_loc:
                print(f"[Exploration] Moving robot from {current_loc} to {target_loc}.")
                obs, reward, done = execute_go(env, task, current_loc, target_loc)
                if done:
                    print("[Exploration] Robot moved to drawer location; terminating early.")
                    return
                robot_location = target_loc
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Try to pick the handle first
        try:
            print(f"[Exploration] Trying to pick handle {handle} on floor.")
            obs, reward, done = execute_pick(env, task, handle, robot_location)
            if done:
                print("[Exploration] Successfully picked up the handle.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Try to pull the drawer -- this action's precondition includes (drawer-closed drawer1)
        try:
            print(f"[Exploration] Attempting to pull open {drawer} using {handle}.")
            obs, reward, done = execute_pull(env, task, drawer, handle, robot_location)
            if done:
                print("[Exploration] Successfully opened the drawer; (drawer-closed drawer1) was likely present.")
            else:
                print("[Exploration] Could not open the drawer; perhaps (drawer-closed drawer1) is not satisfied or missing.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Most likely cause is missing or unsatisfied predicate: (drawer-closed drawer1).")

        # Exploration result log:
        print("[Exploration] If execute_pull failed due to precondition, the missing predicate is likely (drawer-closed drawer1).")
        print("[Feedback] From domain PDDL and feedback: Predicate (drawer-closed drawer1) must be present in the initial state for pull to succeed.")

        # === (Continue with Task Plan Here) ===
        # (Insert further plan steps as needed depending on your oracle plan/goal.)

        # For demonstration, assume we are done after the exploration, as the prompt focuses on missing predicate exploration.
        print("===== Exploration Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()