# run_skeleton_task.py (Completed for Exploration Phase to Find Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # The main focus is to explore the environment to find the missing predicate, 
        # as the planner may be timing out due to missing information about the state (e.g., lock state, weight, durability, etc.).

        # Available skills: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

        # For exploration, we want to gather as much state information as possible about all objects, especially predicates
        # like 'lock-known', 'weight-known', 'durability-known', 'identified', 'temperature-known'.
        # Since only the predefined skills are allowed, we use those in a data-collecting loop.

        # List objects and drawers discovered from the environment
        object_names = [name for name, pos in positions.items() if 'obj' in name or 'object' in name]
        drawer_names = [name for name, pos in positions.items() if 'drawer' in name]
        handle_names = [name for name, pos in positions.items() if 'handle' in name]
        location_names = [name for name, pos in positions.items() if 'location' in name or 'loc' in name or 'table' in name]

        # Try to find robot's current location (using predicates or via observation)
        try:
            robot_location = None
            for name in positions:
                if 'robot' in name and 'location' in name:
                    robot_location = positions[name]
            if not robot_location:
                # fallback to task API or just pick a plausible starting location
                robot_location = next(iter(positions.values()))
        except Exception:
            robot_location = None

        print("[Exploration] Discovered objects:", object_names)
        print("[Exploration] Discovered drawers:", drawer_names)
        print("[Exploration] Discovered handles:", handle_names)
        print("[Exploration] Discovered locations:", location_names)
        print("[Exploration] Robot initial location:", robot_location)

        # The idea: for each object and drawer, try to interact via the available skills and observe what knowledge changes.
        # This "poke and observe" approach helps reveal which predicates are "unknown" before exploration and "known" after using some skill.

        # -- Try to identify objects and explore predicates --
        for obj in object_names:
            # Try picking the object at its location, to see if properties like 'weight-known', 'durability-known' are revealed
            try:
                obj_pos = positions[obj]
                # All locations are used as generic positions here
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to pick {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                    except Exception as e:
                        print(f"[Exploration] Cannot pick {obj} at {loc}: {e}")
            except Exception:
                continue
        
        for drawer in drawer_names:
            # Try to open the drawer by pulling its handle, if a handle is associated
            handle_of_drawer = None
            for hnd in handle_names:
                if drawer in hnd:
                    handle_of_drawer = hnd
                    break
            if not handle_of_drawer and handle_names:
                handle_of_drawer = handle_names[0]
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting pull on {drawer} with handle {handle_of_drawer} at {loc}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_of_drawer, loc)
                except Exception as e:
                    print(f"[Exploration] Cannot pull {drawer} at {loc}: {e}")
            # Try pushing the drawer closed at each location (we don't expect it to succeed if already closed)
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to push {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration] Cannot push {drawer} at {loc}: {e}")

        # Try placing objects into drawers if possible
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] Cannot place {obj} in {drawer} at {loc}: {e}")

        # Try go and sweep motions (may reveal knowledge via sensors)
        # These are less informative unless sensors are present on the robot or locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Robot attempts to go from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                except Exception as e:
                    print(f"[Exploration] Cannot go from {from_loc} to {to_loc}: {e}")
                
                # Try sweeping after going
                for obj in object_names:
                    try:
                        print(f"[Exploration] Attempting to sweep {obj} at {to_loc}")
                        obs, reward, done = execute_sweep(env, task, obj, to_loc)
                    except Exception as e:
                        print(f"[Exploration] Cannot sweep {obj} at {to_loc}: {e}")

        # Try rotate and gripper actions (they may not reveal state but should be explored nonetheless)
        try:
            print("[Exploration] Attempting gripper action")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Gripper action failed: {e}")

        # --- End of exploration phase ---
        print("===== Exploration phase complete. Review logs for evidence of newly revealed predicates (e.g., lock-known, weight-known, durability-known, etc.) =====")
        # Further investigation of the logs would allow a human to determine which predicate is missing
        # in the planning/pddl state, according to which skills are required for successful plan execution.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
