# run_skeleton_task.py (Completed for Missing Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All skills are imported as primitives.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'robot': (x,y,z), 'drawer1': (x1,y1,z1), 'object1': (x2,y2,z2), ...}
        
        # Exploration phase: We need to determine the state of the drawer lock predicate
        # According to the feedback, there is an issue with (drawer-unlocked drawer1)
        # Our exploration goal is to identify the lock state of 'drawer1' (could be locked/unlocked)
        
        # -- Identify main entities --
        # Try to resolve primary names; fallback to first matching keys
        robot_name = None
        drawer_name = None
        handle_name = None
        for k in positions:
            if "robot" in k:
                robot_name = k
            if "drawer" in k and drawer_name is None:
                drawer_name = k
            if "handle" in k and handle_name is None:
                handle_name = k
        
        # Fallbacks if not found
        if robot_name is None:
            robot_name = next(iter(positions.keys()))
        if drawer_name is None:
            drawer_name = [k for k in positions.keys() if 'drawer' in k.lower()]
            if isinstance(drawer_name, list) and len(drawer_name) > 0:
                drawer_name = drawer_name[0]
            else:
                drawer_name = None
        if handle_name is None and drawer_name is not None:
            # Sometimes handles are named e.g., "drawer1_handle"
            for k in positions:
                if "handle" in k and drawer_name.replace('drawer','') in k:
                    handle_name = k
        # Just pick any handle if still not found
        if handle_name is None:
            possible_handles = [k for k in positions if "handle" in k]
            if len(possible_handles) > 0:
                handle_name = possible_handles[0]

        # Get initial robot and object positions
        robot_pos = positions.get(robot_name, None)
        drawer_pos = positions.get(drawer_name, None)
        handle_pos = positions.get(handle_name, None)
        # Fallbacks: If keys missing, skip actual primitive coordinate control (the skill functions may take names)

        # 1. Move robot close to drawer (using execute_go, if available).
        # Simulate exploration to infer the drawer's lock state.
        # We'll try to execute_pull on the drawer; if it fails (because of "locked"), we know the missing predicate.

        exploration_result = None
        plan_success = False

        try:
            # Step 1: Move robot to drawer location
            # Using the 'execute_go' skill: execute_go(from_location, to_location)
            # We'll do this if robot and drawer have positions.

            # First, guess at from/to location names. If none available, use defaults.
            # (Assuming position dict keys can act as location names)
            possible_locs = [k for k in positions if "location" in k or "room" in k or "drawer" in k]
            robot_loc = robot_name
            drawer_loc = drawer_name

            if robot_name != drawer_name:
                try:
                    obs, reward, done = execute_go(env, task, robot_loc, drawer_loc)
                except Exception as e:
                    print(f"[Exploration] Could not execute_go from {robot_loc} to {drawer_loc}: {e}")
            else:
                print("[Exploration] Robot already at drawer location.")

            # Step 2: Pick up the handle (execute_pick skill: execute_pick(object, location))
            try:
                obs, reward, done = execute_pick(env, task, handle_name, drawer_loc)
            except Exception as e:
                print(f"[Exploration] Could not pick handle ({handle_name}) at {drawer_loc}: {e}")

            # Step 3: Try pulling the drawer open (execute_pull skill)
            # execute_pull(drawer, handle, location)
            try:
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_loc)
                plan_success = True
                print(f"[Exploration] Successfully pulled the drawer open (unlocked): predicate 'drawer-unlocked' is set.")
                exploration_result = "drawer-unlocked"
            except Exception as e:
                print(f"[Exploration] Failed to pull the drawer: possibly missing predicate 'drawer-unlocked' or drawer is locked: {e}")
                exploration_result = "drawer-locked"

        except Exception as ex:
            print(f"[Exploration-phase-exception] {ex}")

        # Based on exploration_result, proceed with the oracle plan (here we just demonstrate the predicate exploration)
        # In a full scenario, your plan would now set the appropriate missing predicates for future planning.

        # You can continue the full task execution after exploration as needed using the standard sequence of skills:
        # - execute_pick
        # - execute_place
        # - execute_push
        # etc.
        #
        # For instance:
        # if plan_success:
        #     # Continue to execute pick, place, push, etc...
        #     pass
        # else:
        #     # Try to unlock the drawer, or log failure
        #     pass

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
