# run_skeleton_task.py (Executable code with exploration for predicate discovery and force calibration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including predicate exploration and force calibration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE: Find Missing Predicate/Force Calibration ===
        # The feedback highlighted the need for force calibration and for detecting drawer states/predicates
        # Approach:
        #   1. Try to interact (pull) with drawer handles using varying force if available
        #   2. Observe environment state after each operation (simulate sensory feedback)
        #   3. If pulling action fails or drawer does not open, infer missing predicate (perhaps drawer-locked)
        #   4. Log findings or update your strategy accordingly

        drawer_names = [obj for obj in positions.keys() if "drawer" in obj]
        handle_names = [obj for obj in positions.keys() if "handle" in obj]
        robot_start_location = None
        for k, v in positions.items():
            if "robot" in k:
                robot_start_location = v
                break

        # Assume there's a robot location key in observations, otherwise choose a default location
        robot_location = robot_start_location if robot_start_location is not None else list(positions.values())[0]

        # == 1. Move robot to the drawer location (if not already there) ==
        for drawer in drawer_names:
            drawer_pos = positions[drawer]
            if robot_location != drawer_pos:
                try:
                    # Use execute_go to reach drawer location, if available
                    # Just picking the first two location names for demonstration
                    obs, reward, done = execute_go(
                        env=env,
                        task=task,
                        from_location=robot_location,
                        to_location=drawer_pos
                    )
                    robot_location = drawer_pos
                    print(f"[Exploration] Robot moved to {drawer} at {drawer_pos}")
                except Exception as e:
                    print(f"[Exploration] Exception during move: {e}")

            # == 2. Grasp handle for pull action (must hold the handle object first) ==
            # Check if handle for this drawer exists
            handle = None
            for h in handle_names:
                # simple match: handle-of predicate available in PDDL but here we check names
                if drawer in h:
                    handle = h
                    break
            if handle is None:
                print(f"[Exploration] No handle found for {drawer}")
                continue
            handle_pos = positions[handle]
            try:
                # Pick up handle
                obs, reward, done = execute_pick(env, task, object_name=handle, location=handle_pos)
                print(f"[Exploration] Picked up handle {handle} for {drawer}")
            except Exception as e:
                print(f"[Exploration] Exception during pick(handle): {e}")

            # == 3. Try pull action with standard force calibration ==
            force_options = [0.2, 0.5, 0.8]  # Simulate force calibration: try low, medium, high
            success = False
            for force in force_options:
                try:
                    # Use execute_pull with current force setting, assuming force can be set in API
                    # If skill API doesn't support force, skip this and just call pull
                    print(f"[Exploration] Trying pull on {drawer} with force {force}")
                    obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=drawer_pos, force=force)
                    # After pull, check env state for drawer-open predicate or visually with sensors
                    open_state = False
                    # Check sensor or simulated predicate for drawer open state
                    # For demonstration, try to "sense" with an (imaginary) sensor
                    try:
                        sensor_name = f"{drawer}_open_sensor"
                        sensor = ProximitySensor(sensor_name)
                        detected = sensor.check_collision(Shape(drawer))
                        open_state = detected
                    except Exception:
                        # Fallback: check returned obs for info
                        open_state = 'drawer-open' in str(obs) or 'open' in str(obs)
                    if open_state:
                        print(f"[Exploration] Drawer {drawer} opened successfully with force {force}.")
                        success = True
                        break
                    else:
                        print(f"[Exploration] Drawer {drawer} did NOT open with force {force}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pull with force {force}: {e}")
            if not success:
                print(f"[Exploration] Drawer {drawer} might be locked or force insufficient -- possible missing predicate.")
                # Here, exploration infers the missing predicate, e.g., drawer-locked, and you can plan to unlock/detect lock state

            # Put down the handle to free hand for next exploration
            try:
                obs, reward, done = execute_place(env, task, object_name=handle, drawer=drawer, location=drawer_pos)
                print(f"[Exploration] Placed handle {handle} back in {drawer}")
            except Exception:
                # If place fails, might be because only pick/place allowed for some types -- ignore for exploration
                pass

        # === TASK EXECUTION PHASE: Use only available/predefined skills in plan logic ===

        # -- Example Plan: Open drawer, pick item from floor, put in the drawer, push drawer closed --

        # Assume existence of at least one object on the floor and one openable drawer
        floor_objects = [obj for obj in positions.keys() if "object" in obj and "floor" in positions[obj]]
        for drawer in drawer_names:
            drawer_pos = positions[drawer]
            # Move to the drawer
            if robot_location != drawer_pos:
                try:
                    obs, reward, done = execute_go(env=env, task=task, from_location=robot_location, to_location=drawer_pos)
                    robot_location = drawer_pos
                except Exception as e:
                    print(f"[Task] Exception during move to drawer: {e}")
            # Find an object to pick
            if not floor_objects:
                print("[Task] No floor objects found for manipulation.")
                break
            obj = floor_objects[0]
            obj_pos = positions[obj]
            # Pick object from floor
            try:
                obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_pos)
                print(f"[Task] Picked up object {obj} from {obj_pos}")
            except Exception as e:
                print(f"[Task] Exception during object pick: {e}")
            # Open the drawer (assume handle already available from previous, or pick handle if necessary)
            handle = None
            for h in handle_names:
                if drawer in h:
                    handle = h
                    break
            if handle:
                handle_pos = positions[handle]
                try:
                    obs, reward, done = execute_pick(env, task, object_name=handle, location=handle_pos)
                except Exception:
                    pass
                try:
                    obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=drawer_pos)
                    print(f"[Task] Pulled handle {handle} to open {drawer}")
                except Exception as e:
                    print(f"[Task] Exception during pull (open drawer): {e}")
            # Place object in drawer
            try:
                obs, reward, done = execute_place(env, task, object_name=obj, drawer=drawer, location=drawer_pos)
                print(f"[Task] Placed object {obj} into {drawer}")
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
            # Push (close) the drawer
            try:
                obs, reward, done = execute_push(env, task, drawer=drawer, location=drawer_pos)
                print(f"[Task] Pushed closed {drawer}")
            except Exception as e:
                print(f"[Task] Exception during push: {e}")
            # Done -- simple demo loop (remove object from floor_objects so it only runs once)
            floor_objects.pop(0)
            break

        print("===== Skeleton Task Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
